/*
 * (c) CBD BC, Russia, Spb.
 *
 * Mail: support@kpda.ru
 *
 * Date: 07/10/2010
 * Dev:  A. Docuchaev
 */


/* main.c */


#include <ioadm.h>
#include <piso725_msg.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <devctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <sys/dispatch.h>


ioadm_device_t              piso725 = IOADM_DEVICE_INIT;
char                        device[7] = "piso725";
int                         pci_index = 0;


void parse_options( int argc, char *argv[] )
{
    int                 c;

    while (-1 != (c = getopt(argc, argv, "i:"))) {
        switch (c) {
            case 'i':
                pci_index = strtoul( optarg, 0, 0 );
                break;
            default:
                printf( "Error: Undefined option \"-%c\"\n", c );
                exit( -1 );
        }
    }
}

/*
 * PISO-725 D-type connecion (isolated digital input DI0):
 *   TTL[+5]:  ----------> pin 12 (DIA0)
 *   TTL[GND]: ----------> pin 29 (GND)
 *                   `---> pin 2  (COM0)
 *                    ---> pin 1  (NO0)
 *                    `--> pin 30 (DIB0)
 */
int main( int argc, char *argv[] )
{
    int                     i;
    int                     sz;
    uint32_t                rdata;
    struct sched_param      sched_param;
    unsigned long long      tmp;
    piso725_line_t          oline = 0x80;

    parse_options( argc, argv );

    sched_param.sched_priority = 70;
    pthread_setschedparam( pthread_self(), SCHED_FIFO, &sched_param );

    /* Try to open device piso725 */
    if ( ioadm_open_pci( &piso725, device, pci_index ) ) {
        printf( "Error: can't open device\n" );
        exit( -1 );
    }

    /* Prepare lib to I/O operations */
    if ( ioadm_lib_prepare_io( &piso725 ) != 0 ) {
        printf( "Error: can't prepare lib-i/o\n" );
        exit( -1 );
    }

    /* Enable EXTIRQ source */
    if ( ioadm_set_irq_sources( &piso725, PISO725_IRQ_SOURCE_DI0 ) != 0 ) {
        printf( "Error: can't activate DI0IRQ\n" );
        exit( -1 );
    }

    /* Set positive EXTIRQ polarity (all other IRQ sources set to negative polarity value) */
    if ( ioadm_set_irq_polarity( &piso725, PISO725_IRQ_POSITIVE_POLARITY_DI0 ) != 0 ) {
        printf( "Error: can't set DI0IRQ polarity\n" );
        exit( -1 );
    }

    /* Start piso725 */
    if ( ioadm_start( &piso725, 0 ) != 0 ) {
        printf( "Error: can't start device\n" );
        exit( -1 );
    }

    ioadm_set_sample( &piso725, &oline, sizeof( oline ) );

    /* Read data (1) */
    for ( i = 0; i < 10; i++ ) {

        /* Waiting for EXTIRQ event */
        if ( ioadm_wait_irq_event( &piso725 ) != 0 ) {
            printf( "Error: can't identify IRQ event\n" );
            exit( -1 );
        }
        sz = ioadm_read( &piso725, &rdata, sizeof( rdata ) );

    }
    tmp = ioadm_get_readed_data_sz( &piso725 );
    printf( "1. Readed data size:                      %lldb\n", tmp );
    printf( "   Read cycles:                           %lld\n",  tmp );

    /* Read data (2) */
    ioadm_clear_statistic( &piso725 );
    sleep( 5 );
    tmp = ioadm_get_received_data_sz( &piso725 );
    printf( "2. Received data size (with 5 sec delay): %lldb\n", tmp );
    printf( "   IRQ count (with 5 sec delay):          %lld\n",  tmp );

    printf( "3. Time between 2 interrupts:             %f sec\n", ioadm_get_b2int_time( &piso725 ) );
    printf( "4. Read time (1 byte):                    %f sec\n", ioadm_get_read_time( &piso725 ) );
    printf( "   Read-back time (1 byte):               %f sec\n", ioadm_get_readback_time( &piso725 ) );
    printf( "   Write time (1 byte):                   %f sec\n", ioadm_get_write_time( &piso725 ) );

    ioadm_set_sample( &piso725, &oline, sizeof( oline ) );

    /* Stop piso725 */
    if ( ioadm_stop( &piso725, 0 ) != 0 ) {
        printf( "Error: can't start device\n" );
        exit( -1 );
    }

    ioadm_close( &piso725 );

    return (0);
}
