/*
 * (c) CBD BC, Russia, Spb.
 *
 * Mail: support@kpda.ru
 *
 * Date: 07/10/2010
 * Dev:  A. Docuchaev
 */


/* piso.h */


#include <io-adm-drv.h>
#include <io-adm-msg.h>
#include <ioadm.h>
#include <piso725_msg.h>

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <malloc.h>
#include <unistd.h>
#include <devctl.h>
#include <fcntl.h>
#include <sys/neutrino.h>
#include <sys/syspage.h>
#include <sys/mman.h>
#include <hw/pci.h>
#include <hw/inout.h>


#ifndef _CBD_BC_IO_ADM_PISO725_H_
#define _CBD_BC_IO_ADM_PISO725_H_


/* Defs */
#define MSG                             fprintf
#define ERRF                            stderr
#define STDF                            stdout

#define PCI_VENDOR_ID_ICPDAS            0xE159
#define PCI_DEVICE_ID_N0                0x2
#define PCI_DEVICE_ID_N1                0x1
#define PCI_SUBDEVICE_ID_PISO725R2      0x00000c0080ull
#define PCI_SUBDEVICE_ID_PISO725R21     0x0000000380ull
#define PCI_SUBDEVICE_ID_PISO725R22     0x0000004380ull
#define PCI_SUBDEVICE_ID_PISO725R23     0x000000c380ull
#define PCI_DEVICE_NAME_PISO725R2       "ICP DAS PISO-725 (rev.2.0)"
#define PCI_DEVICE_NAME_PISO725R21      "ICP DAS PISO-725 (rev.2.1)"
#define PCI_DEVICE_NAME_PISO725R22      "ICP DAS PISO-725 (rev.2.2)"
#define PCI_DEVICE_NAME_PISO725R23      "ICP DAS PISO-725 (rev.2.3)"

#define PCI_SCAN_DEPTH                  32
#define MAX_PISO725_DEVICE_COUNT        32

/* Registers */
#define PISO725_RESET_CONTROL           0x00
#define PISO725_INT_MASK_CONTROL        0x05
#define PISO725_AUX_PIN_STATUS          0x07
#define PISO725_INT_POLARITY            0x2a
#define PISO725_DO                      0xc0
#define PISO725_DI                      0xc4


/* Types */

typedef struct piso725_dev {

    io_adm_drv_descriptor_t     descriptor; /* MUST BE FIRST */

    /* Device */
    uint32_t            sub_id0;            /* Sub AUX */
    uint32_t            sub_id1;            /* Sub vid:did */
    char                name[65];           /* Device name */

    /* PCI */
    int                 pci_server;
    void                *device;
    int                 attached;
    uint16_t            vid;
    uint16_t            did;
    int                 idx;                /* PCI index */

    /* I/O */
    uint32_t            base;               /* Base address */

    /* IRQ */
    uint32_t            irq;
    int                 iid;                /* Interrupt ID */
    struct sigevent     irq_notifier;
    uint8_t             int_mask;           /* current PISO725_INT_MASK_CONTROL register value */
    uint8_t             int_polarity;       /* current PISO725_INT_POLARITY register value */
    uint8_t             irq_source,         /*
                                             * xxxxxxxx
                                             * |||||||`--- DI0: 1/0
                                             * ||||||`---- DI1: 1/0
                                             * |||||`----- DI2: 1/0
                                             * ||||`------ DI3: 1/0
                                             * |||`------- DI4: 1/0
                                             * ||`-------- DI5: 1/0
                                             * |`--------- DI6: 1/0
                                             * `---------- DI7: 1/0
                                             */
                        irq_edge;           /*
                                             * xxxxxxxx
                                             * |||||||`--- DI0: 1 - positive, 0 - negative
                                             * ||||||`---- DI1: 1 - positive, 0 - negative
                                             * |||||`----- DI2: 1 - positive, 0 - negative
                                             * ||||`------ DI3: 1 - positive, 0 - negative
                                             * |||`------- DI4: 1 - positive, 0 - negative
                                             * ||`-------- DI5: 1 - positive, 0 - negative
                                             * |`--------- DI6: 1 - positive, 0 - negative
                                             * `---------- DI7: 1 - positive, 0 - negative
                                             */

    /* Time parameters */
    volatile uint64_t   int_st_cc;          /* Interrupt start clock-cycles */
    volatile uint64_t   int_ed_cc;          /* Interrupt end clock-cycles */
    volatile uint64_t   int_iv_cc;          /* Interrupt interval clock-cycles */
    double              int_time;           /* Interrupt time (in sec) */
    uint64_t            read_iv_cc;         /* Read interval (in clock-cycles) */
    double              read_time;          /* Read interval (in sec) */
    uint64_t            write_iv_cc;        /* Write interval (in clock-cycles) */
    double              write_time;         /* Write interval (in sec) */

    /* Misc */
    int                         verbose;
    int                         busscan;    /* BUS scanning mode flag */
    uint64_t                    irqsources; /* User selected IRQs (see DCMD_IOADM_SET_IRQ_SOURCE) */
    uint64_t                    irqpolar;   /* User selected IRQs (see DCMD_IOADM_SET_IRQ_POLARITY) */

} __attribute__ ((__packed__)) piso725_dev_t;


/* Funcs */

/* init.c */
int dev_parse_options( piso725_dev_t *dev, char *options );
void * io_adm_drv_init( char *options );
uint8_t io_adm_drv_busscan_result( void );
char * io_adm_drv_devname( void *device );
uint32_t io_adm_drv_devid( void *device );
uint64_t io_adm_drv_devsubid( void *device );
void * io_adm_drv_devstat( void *device );
int io_adm_drv_open( void *device );
void io_adm_drv_close( void *device );
void io_adm_drv_destroy( void *device );

/* io.c */
int io_adm_drv_read( void *device, void *msg, uint32_t size );
int io_adm_drv_write( void *device, void *msg, uint32_t size );
int io_adm_drv_devctl( void *device, io_devctl_t *msg, void *data );

/* event.c */
void init_isr( piso725_dev_t *device );
const struct sigevent * irq_handler( void *area, int id );

/* busscan.c */
int piso725_busscan( piso725_dev_t *dev, uint16_t vid, uint16_t did, int pci_id );
void piso725_busscan_detach( piso725_dev_t *dev );
int piso725_busscan_attached( uint16_t vid, uint16_t did, int idx );
int piso725_busscan_result();

/* in.c */
uint8_t piso725_read8( piso725_dev_t *dev, uint8_t iline );
uint8_t piso725_readback8( piso725_dev_t *dev, uint8_t oline );
uint8_t piso725_read1( piso725_dev_t *dev, uint8_t iline );

/* out.c */
void piso725_write8( piso725_dev_t *dev, uint8_t oline, uint8_t val );


#endif  /* _CBD_BC_IO_ADM_PISO725_H_ */
