/*
 * (c) CBD BC, Russia, Spb.
 *
 * Mail: support@kpda.ru
 *
 * Date: 04/10/2010
 * Dev:  A. Docuchaev
 */


/* io.c */


#include "pio.h"


/* Defines */

#define DEVCTL_RET( cond, devctl_status )       \
    if ( cond ) {                               \
        msg->o.ret_val = (devctl_status);       \
        break;                                  \
    }


/* Funcs */

/*
 * _IO_READ handler
 *
 * In: Device ptr
 *     Message ptr (Message size >= 4)
 *     Size to read
 * Out: n - count of readed bytes
 *      0 - error
 */
int io_adm_drv_read( void *device, void *msg, uint32_t size )
{
    piod64_dev_t            *dev    = (piod64_dev_t *)device;
    int                     read_sz = 0;

    /* Device not started */
    if ( !(dev->descriptor.device_state & IO_ADM_DRV_STATE_DEV_STARTED) )
        return (0);

    /* Process */
    if ( size >= 4 ) {
        *((uint32_t *)msg) = piod64_read32( dev );
        read_sz = 4;
    } else
    if ( size >= 2 ) {
        *((uint16_t *)msg) = piod64_read16( dev );
        read_sz = 2;
    } else
    if ( size >= 1 ) {
        *((uint8_t *)msg) = piod64_read8( dev, 1 );
        read_sz = 1;
    }

    dev->descriptor.data_readed += read_sz;
    return (read_sz);
}

/*
 * _IO_WRITE handler
 *
 * In: Device ptr
 *     Message ptr
 *     Message size
 * Out: -1 - error
 */
int io_adm_drv_write( void *device, void *msg, uint32_t size )
{
    piod64_dev_t            *dev     = (piod64_dev_t *)device;
    int                     write_sz = 0;

    /* Device not started */
    if ( !(dev->descriptor.device_state & IO_ADM_DRV_STATE_DEV_STARTED) )
        return (-1);

    /* Process */
    if ( size == 4 ) {
        piod64_write32( dev, *((uint32_t *)msg) );
        write_sz = 4;
    } else
    if ( size >= 2 ) {
        piod64_write16( dev, *((uint16_t *)msg) );
        write_sz = 2;
    } else
    if ( size == 1 ) {
        piod64_write8( dev, 1, *((uint8_t *)msg) );
        write_sz = 1;
    }

    dev->descriptor.data_transmitted += write_sz;
    return (write_sz);
}

/*
 * _IO_DEVCTL handler
 *
 * In: Device ptr
 *     io_devctl_t message ptr
 *     Buffer ptr
 * Out: 0 - ok
 */
int io_adm_drv_devctl( void *device, io_devctl_t *msg, void *data )
{
    piod64_dev_t            *dev = (piod64_dev_t *)device;

    switch ( msg->i.dcmd ) {

        case DCMD_IOADM_GET_BUFF_MODE: {
            /* Process DCMD_IOADM_GET_BUFF_MODE */
            msg->o.ret_val = LIBMODE_SIMPLE | LIBMODE_BUFPERM_R | LIBMODE_BUFPERM_W;
            break;
        }

        case DCMD_IOADM_START: {
            /* Enable device */
            dev->descriptor.device_state |= IO_ADM_DRV_STATE_DEV_STARTED;
            out8( dev->base + PIO_D64_RESET_CONTROL, 0x1 );

            /* Enable IRQ */
            dev->descriptor.device_state |= IO_ADM_DRV_STATE_IRQ_ENABLED;
            out8( dev->base + PIO_D64_INT_MASK_CONTROL, dev->irqsources & 0x7 );
            dev->int_mask = in8( dev->base + PIO_D64_INT_MASK_CONTROL ) & 0x0f;
            /* Set positive IRQ polarity */
            out8( dev->base + PIO_D64_INT_POLARITY, dev->irqpolar & 0x7 );
            dev->int_polarity = in8( dev->base + PIO_D64_INT_POLARITY ) & 0x0f;

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_STOP: {
            /* Disable device */
            out8( dev->base + PIO_D64_RESET_CONTROL,    0x0 );
            dev->descriptor.device_state &= ~IO_ADM_DRV_STATE_DEV_STARTED;

            /* Disable IRQ */
            out8( dev->base + PIO_D64_INT_MASK_CONTROL, 0x0 );
            dev->descriptor.device_state &= ~IO_ADM_DRV_STATE_IRQ_ENABLED;
            dev->int_mask = in8( dev->base + PIO_D64_INT_MASK_CONTROL ) & 0x0f;

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_PAUSE: {
            /* Disable IRQ */
            out8( dev->base + PIO_D64_INT_MASK_CONTROL, 0x0 );
            dev->descriptor.device_state &= ~IO_ADM_DRV_STATE_IRQ_ENABLED;
            dev->int_mask = in8( dev->base + PIO_D64_INT_MASK_CONTROL ) & 0x0f;

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_RESUME: {
            /* Enable IRQ */
            dev->descriptor.device_state |= IO_ADM_DRV_STATE_IRQ_ENABLED;
            out8( dev->base + PIO_D64_INT_MASK_CONTROL, dev->irqsources & 0x7 );
            dev->int_mask = in8( dev->base + PIO_D64_INT_MASK_CONTROL ) & 0x0f;

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_SET_IRQ_SOURCE: {
            if ( dev->descriptor.device_state & IO_ADM_DRV_STATE_DEV_STARTED ||
                 dev->descriptor.device_state & IO_ADM_DRV_STATE_IRQ_ENABLED ) {
                msg->o.nbytes  = 0;
                msg->o.ret_val = 1;
                break;
            }
            dev->irqsources = *(uint32_t *)data;
            /* This value will be processed here -> DCMD_IOADM_START */

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_SET_IRQ_POLARITY: {
            if ( dev->descriptor.device_state & IO_ADM_DRV_STATE_DEV_STARTED ||
                 dev->descriptor.device_state & IO_ADM_DRV_STATE_IRQ_ENABLED ) {
                msg->o.nbytes  = 0;
                msg->o.ret_val = 1;
                break;
            }
            dev->irqpolar = *(uint32_t *)data;
            /* This value will be processed here -> DCMD_IOADM_START */

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_WAIT_IRQ_EV: {
            if ( !(dev->descriptor.device_state & IO_ADM_DRV_STATE_DEV_STARTED) ||
                 !(dev->descriptor.device_state & IO_ADM_DRV_STATE_IRQ_ENABLED) ) {
                msg->o.nbytes  = 0;
                msg->o.ret_val = 1;
                break;
            }

            InterruptWait( 0, NULL );

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_SAMPLE: {
            piod64_line_t       *line   = (piod64_line_t *)data;

            DEVCTL_RET( *line >= 32, 1 )
            DEVCTL_RET( *line < 0,   1 )

            *line = piod64_read1( dev, *line );

            msg->o.nbytes  = sizeof( piod64_line_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_SET_SAMPLE: {
            piod64_line_t       *line   = (piod64_line_t *)data;

            DEVCTL_RET( (*line & 0x7f) >= 32, 1 )
            DEVCTL_RET( (*line & 0x7f) < 0,   1 )

            piod64_write8( dev, (*line & 0x7f), *line >> 7 );

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_AVAIL_DATA_SZ: {
            if ( dev->descriptor.device_state & IO_ADM_DRV_STATE_DEV_STARTED )
                *(uint32_t *)data = 4 /* 32 line * 1 bit = 4 bytes */;
            else
                *(uint32_t *)data = 0 /* I/O operations not permitted by th hardware (see DCMD_IOADM_START) */;

            msg->o.nbytes  = sizeof( uint32_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_LAST_RECV_TIME: {
            *(uint64_t *)data = dev->int_ed_cc;

            msg->o.nbytes  = sizeof( uint64_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_INT_TIME: {
            dev->int_time = (double)(dev->int_iv_cc) /
                                    (SYSPAGE_ENTRY(qtime)->cycles_per_sec);

            *(double *)data = dev->int_time;

            msg->o.nbytes  = sizeof( double );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_RD_TIME: {
            if ( !(dev->descriptor.device_state & IO_ADM_DRV_STATE_DEV_STARTED) ) {
                msg->o.nbytes  = 0;
                msg->o.ret_val = -1;
                break;
            }

            /* Get read time */
            dev->read_iv_cc = ClockCycles();
            piod64_read32( dev );
            dev->read_iv_cc = ClockCycles() - dev->read_iv_cc;

            dev->read_time = (double)(dev->read_iv_cc) /
                                     (SYSPAGE_ENTRY(qtime)->cycles_per_sec);

            *(double *)data = dev->read_time;

            msg->o.nbytes  = sizeof( double );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_WR_TIME: {
            if ( !(dev->descriptor.device_state & IO_ADM_DRV_STATE_DEV_STARTED) ) {
                msg->o.nbytes  = 0;
                msg->o.ret_val = -1;
                break;
            }

            /* Get read time */
            dev->write_iv_cc = ClockCycles();
            piod64_write32( dev, 0xffffffff );
            dev->write_iv_cc = ClockCycles() - dev->write_iv_cc;

            dev->write_time = (double)(dev->write_iv_cc) /
                                     (SYSPAGE_ENTRY(qtime)->cycles_per_sec);

            *(double *)data = dev->write_time;

            msg->o.nbytes  = sizeof( double );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_STATE: {
            *(uint32_t *)data = dev->descriptor.device_state;

            msg->o.nbytes  = sizeof( uint32_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_RECEIVED_DATA_SZ: {
            *(uint64_t *)data = dev->descriptor.data_received;

            msg->o.nbytes  = sizeof( uint64_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_READED_DATA_SZ: {
            *(uint64_t *)data = dev->descriptor.data_readed;

            msg->o.nbytes  = sizeof( uint64_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_CLEAR_STATISTIC: {
            IO_ADM_DRV_DESCRIPTOR_CLEAR_STATS( dev->descriptor )

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_READ_REG: {
            if ( ((ioadm_reg_desc_t *)data)->reg_type != PIOD64_REG_TYPE_PCI ) {
                msg->o.nbytes  = 0;
                msg->o.ret_val = -1;
                break;
            }

            if ( (((ioadm_reg_desc_t *)data)->reg_offset == PIO_D64_INT_POLARITY) ||
                 (((ioadm_reg_desc_t *)data)->reg_offset == PIO_D64_INT_MASK_CONTROL) ) {
                /* Access to this regs denied */
                msg->o.nbytes  = 0;
                msg->o.ret_val = -2;
                break;
            }

            ((ioadm_reg_desc_t *)data)->reg_value = in8( dev->base + (uint32_t)(((ioadm_reg_desc_t *)data)->reg_offset) );

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_WRITE_REG: {
            if ( ((ioadm_reg_desc_t *)data)->reg_type != PIOD64_REG_TYPE_PCI ) {
                msg->o.nbytes  = 0;
                msg->o.ret_val = -1;
                break;
            }

            if ( (((ioadm_reg_desc_t *)data)->reg_offset == PIO_D64_INT_POLARITY) ||
                 (((ioadm_reg_desc_t *)data)->reg_offset == PIO_D64_INT_MASK_CONTROL) ) {
                /* Access to this regs denied */
                msg->o.nbytes  = 0;
                msg->o.ret_val = -2;
                break;
            }

            out8( dev->base + (uint32_t)(((ioadm_reg_desc_t *)data)->reg_offset),
                  (uint8_t)(((ioadm_reg_desc_t *)data)->reg_value) );

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        default:
            msg->o.ret_val = 0xff;  /* Error status */
            msg->o.nbytes  = 0;

    }

    return (0);
}
