#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <Pt.h>
#include <time.h>

#include <ioadm.h>
#include <l783m_msg.h>


ioadm_device_t              l783m = IOADM_DEVICE_INIT;
static PtWidget_t           *feed_toggle_wgt,
                            *mtrend_wgt,
                            *label1,
                            *label2,
                            *label3;
int                         mode = 0;       /* Modes: 0 - Single sample reads
                                                      1 - Buffer reads */
int                         pci_index = 0;
int                         print_adc_samples = 0;


/* Draw ADC graphic */
void Draw( int mode )
{
    int                         i      = 0;
    uint16_t                    *ptr   = NULL;
    l783m_msg_sample_ranges_t   sr;

    /* Get sample ranges */
    sr.gain = 0;
    if ( ioadm_get_sample_ranges( &l783m, (void *)&sr, sizeof(l783m_msg_sample_ranges_t) ) != 0 ) {
        printf( "Error: can't get sample ranges\n" );
        exit( -1 );
    }

    if ( mode ) {

        int                 sz     = 20;
        PtArg_t             args[2];
        char                text[30];
        struct timespec     start, stop;
        double              interval;

        clock_gettime( CLOCK_REALTIME, &start );

        /* Start ADC */
        if ( ioadm_start( &l783m, 0 ) != 0 ) {
            printf( "Error: can't start ADC\n" );
            exit( -1 );
        }

        sz = ioadm_read_fragment_wc( &l783m, (void *)&ptr );

        clock_gettime( CLOCK_REALTIME, &stop );

        /* Stop ADC */
        if ( ioadm_stop( &l783m, 0 ) != 0 ) {
            printf( "Error: can't start ADC\n" );
            exit( -1 );
        }

        /* Set conversion time */
        interval = (stop.tv_sec - start.tv_sec) + 
                   (double)(stop.tv_nsec - start.tv_nsec) / (double)(1000000000L);
        sprintf( text, "%3.1f", interval * 1000 );
        PtSetArg( &args[0], Pt_ARG_TEXT_STRING, text, 0 );
        PtSetResources( label1, 1, args );

        if ( sz > 0 ) {

            int             min    = 0;
            int             max    = 0;

            for ( i = 0; i < sz / 2; i++ ) {

                double scale;
                int x, c, cu, u;

                scale = (sr.u_max - sr.u_min);
                scale /= (sr.sample_val_max - sr.sample_val_min);
                c = (sr.sample_val_min + sr.sample_val_max) / 2;
                cu = (sr.u_min + sr.u_max) / 2;

                x = (int)(short int)ptr[i];

                u = (x - c) * scale + cu;
                u = (double)u / (double)1000;

                if ( u < min )
                    min = u;
                if ( u > max )
                    max = u;

                PtMTrendAddData( mtrend_wgt, 0, &u, 1 );

            }

            /* Set min/max values */
            sprintf( text, "%i", min );
            PtSetArg( &args[0], Pt_ARG_TEXT_STRING, text, 0 );
            PtSetResources( label3, 1, args );
            sprintf( text, "%i", max );
            PtSetArg( &args[0], Pt_ARG_TEXT_STRING, text, 0 );
            PtSetResources( label2, 1, args );

        }

        /* Stop timer */
        long                *f;
        PtGetResource( feed_toggle_wgt, Pt_ARG_FLAGS, &f, 0 );
        *f &= ~(unsigned long)Pt_SET;
        PtSetResource( feed_toggle_wgt, Pt_ARG_FLAGS, *f, 0 );
        PtReRealizeWidget( feed_toggle_wgt );

        //printf( ">>>>>>>>>> IntT=%f\n", ioadm_get_b2int_time( &l783m ) );

    } else {

        int                 sz     = 20;

        for ( i = 0; i < sz / 2; i++ ) {

            l783m_msg_channel_t     ch_msg;
            ch_msg.index = 0;
            ch_msg.mode  = L783M_CHANNEL_SINGLE_ENDED;
            ch_msg.input = 0;
            ch_msg.gain  = 0;

            if ( (sz = ioadm_get_sample( &l783m, &ch_msg, sizeof(ch_msg) )) > 0 ) {
                double scale;
                int x, c, cu, u;

                scale = (sr.u_max - sr.u_min);
                scale /= (sr.sample_val_max - sr.sample_val_min);
                c = (sr.sample_val_min + sr.sample_val_max) / 2;
                cu = (sr.u_min + sr.u_max) / 2;

                x = (int)(signed short)sz;

                u = (x - c) * scale + cu;
                u = (double)u / (double)1000;

                PtMTrendAddData( mtrend_wgt, 0, &u, 1 );

                if ( print_adc_samples )
                    printf("0x%.4x (%i mV)\n", x & 0xffff, u );
            }
        }

    }
}


/* Timer callback */
int timer_func( PtWidget_t *widget, void *data, PtCallbackInfo_t *cbinfo )
{
    if ( Pt_SET & PtWidgetFlags( feed_toggle_wgt ) ) {
        Draw( mode );
    }

    return Pt_CONTINUE;
}


/* Initialize ioadm lib */
void init_ioadm_lib( int mode, int pci_index )
{
    char                    device[6] = "l783m";
    l783m_msg_channel_t     ch_msg;
    l783m_buff_info_t       bi;

    /* Try to open device l783m */
    if ( ioadm_open_pci( &l783m, device, pci_index ) ) {
        printf( "Error: can't open device\n" );
        exit( -1 );
    }

    if ( !mode )
        return;

    /* Set channels number */
    if ( ioadm_set_channels_number( &l783m, 1/*8*/ ) ) {
        printf( "Error: can't set channel number\n" );
        exit( -1 );
    }

    /* Set channels */
    ch_msg.index = 0;
    ch_msg.mode  = L783M_CHANNEL_SINGLE_ENDED;
    ch_msg.input = 0;
    ch_msg.gain  = 0;
    if ( ioadm_custom_dcmd( &l783m, DCMD_IOADM_SET_CHANNEL, (void *)&ch_msg, sizeof( ch_msg ) ) != 0 ) {
        printf( "Error: can't set channel's metric\n" );
        exit( -1 );
    }
/*    ch_msg.index = 1;
    ch_msg.mode  = L783M_CHANNEL_SINGLE_ENDED;
    ch_msg.input = 1;
    ch_msg.gain  = 1;
    if ( ioadm_custom_dcmd( &l783m, DCMD_IOADM_SET_CHANNEL, (void *)&ch_msg, sizeof( ch_msg ) ) != 0 ) {
        printf( "Error: can't set channel's metric\n" );
        exit( -1 );
    }
    ch_msg.index = 2;
    ch_msg.mode  = L783M_CHANNEL_SINGLE_ENDED;
    ch_msg.input = 2;
    ch_msg.gain  = 1;
    if ( ioadm_custom_dcmd( &l783m, DCMD_IOADM_SET_CHANNEL, (void *)&ch_msg, sizeof( ch_msg ) ) != 0 ) {
        printf( "Error: can't set channel's metric\n" );
        exit( -1 );
    }
    ch_msg.index = 3;
    ch_msg.mode  = L783M_CHANNEL_SINGLE_ENDED;
    ch_msg.input = 3;
    ch_msg.gain  = 1;
    if ( ioadm_custom_dcmd( &l783m, DCMD_IOADM_SET_CHANNEL, (void *)&ch_msg, sizeof( ch_msg ) ) != 0 ) {
        printf( "Error: can't set channel's metric\n" );
        exit( -1 );
    }
    ch_msg.index = 4;
    ch_msg.mode  = L783M_CHANNEL_SINGLE_ENDED;
    ch_msg.input = 4;
    ch_msg.gain  = 1;
    if ( ioadm_custom_dcmd( &l783m, DCMD_IOADM_SET_CHANNEL, (void *)&ch_msg, sizeof( ch_msg ) ) != 0 ) {
        printf( "Error: can't set channel's metric\n" );
        exit( -1 );
    }
    ch_msg.index = 5;
    ch_msg.mode  = L783M_CHANNEL_SINGLE_ENDED;
    ch_msg.input = 5;
    ch_msg.gain  = 1;
    if ( ioadm_custom_dcmd( &l783m, DCMD_IOADM_SET_CHANNEL, (void *)&ch_msg, sizeof( ch_msg ) ) != 0 ) {
        printf( "Error: can't set channel's metric\n" );
        exit( -1 );
    }
    ch_msg.index = 6;
    ch_msg.mode  = L783M_CHANNEL_SINGLE_ENDED;
    ch_msg.input = 6;
    ch_msg.gain  = 1;
    if ( ioadm_custom_dcmd( &l783m, DCMD_IOADM_SET_CHANNEL, (void *)&ch_msg, sizeof( ch_msg ) ) != 0 ) {
        printf( "Error: can't set channel's metric\n" );
        exit( -1 );
    }
    ch_msg.index = 7;
    ch_msg.mode  = L783M_CHANNEL_SINGLE_ENDED;
    ch_msg.input = 7;
    ch_msg.gain  = 1;
    if ( ioadm_custom_dcmd( &l783m, DCMD_IOADM_SET_CHANNEL, (void *)&ch_msg, sizeof( ch_msg ) ) != 0 ) {
        printf( "Error: can't set channel's metric\n" );
        exit( -1 );
    }*/

    /* Set buffer size */
    bi.frag_sz = /*0x200*/ 8 * 350;
    bi.buf_sz  = /*0x20000*/0;
    if ( ioadm_set_buff( &l783m, (void *)&bi, sizeof( bi ) ) != 0 ) {
        printf( "Error: can't set buffer\n" );
        exit( -1 );
    }

    /* Set frame rate */
    /*if ( ioadm_set_frame_rate( &l783m, 800000 ) != 0 ) {
        printf( "Error: can't set frame rate\n" );
        exit( -1 );
    }*/
    {
        l783m_msg_timing_params_t   tp;
            tp.channel_rate = 8 * 100 * 1000;
            tp.frame_rate   = 0;

        if ( ioadm_set_timing( &l783m, &tp, sizeof( tp ) ) != 0 ) {
            printf( "Error: can't set timing parameters\n" );
            exit( -1 );
        }
    }

    /* Set calibration multipliers */
    if ( ioadm_set_calibration_multipliers( &l783m, NULL, 0 ) != 0 ) {
        printf( "Error: can't set calibration multipliers\n" );
        exit( -1 );
    }

    /* Enable calibration */
    if ( ioadm_enable_calibration( &l783m, 1 ) != 0 ) {
        printf( "Error: can't enable calibration\n" );
        exit( -1 );
    }

    /* Prepare lib to I/O operations */
    if ( ioadm_lib_prepare_io( &l783m ) != 0 ) {
        printf( "Error: can't prepare lib-i/o\n" );
        exit( -1 );
    }
}


/* Command line parser */
void parse_options( int argc, char *argv[] )
{
    int                 c;

    while (-1 != (c = getopt(argc, argv, "i:fs"))) {
        switch (c) {
            case 'i':
                pci_index = strtoul( optarg, 0, 0 );
                break;
            case 'f':
                mode = 1;
                break;
            case 's':
                print_adc_samples = 1;
                break;
            default:
                printf( "Error: Undefined option \"-%c\"\n", c );
                exit( -1 );
        }
    }
}


int main( int argc, char *argv[] )
{
    PtWidget_t              *window;
    PtArg_t                 args[20];
    int                     i;

    parse_options( argc, argv );

    /* Application window */
    i = 0;
    PtSetArg( &args[i++], Pt_ARG_WINDOW_TITLE, "MTrend Sample Application", 0 );
    if ( !mode ) {
        PtSetArg( &args[i++], Pt_ARG_HEIGHT, 400, 0 );
        PtSetArg( &args[i++], Pt_ARG_WIDTH,  600, 0 );
    } else {
        PtSetArg( &args[i++], Pt_ARG_HEIGHT, 400, 0 );
        PtSetArg( &args[i++], Pt_ARG_WIDTH,  800, 0 );
    }

    if( NULL == ( window = PtAppInit( NULL, &argc, argv, i, args ) ) ) {
        perror( "PtAppInit()" );
        return 1;
    }

    /* Labels */
    if ( mode ) {
        /* 1 */
        {
        PhArea_t        area = { {600, 30}, {50, 20} };

        i = 0;
        PtSetArg( &args[i++], Pt_ARG_TEXT_STRING, "Conversion time (ms.):", 0 );
        PtSetArg( &args[i++], Pt_ARG_AREA, &area, 0 );
        PtCreateWidget( PtLabel, NULL, i, args );
        }
        /* 2 */
        {
        PhArea_t        area = { {750, 30}, {50, 20} };

        i = 0;
        PtSetArg( &args[i++], Pt_ARG_TEXT_STRING, "0", 0 );
        PtSetArg( &args[i++], Pt_ARG_AREA, &area, 0 );
        label1 = PtCreateWidget( PtLabel, NULL, i, args );
        }
        /* 3 */
        {
        PhArea_t        area = { {600, 50}, {50, 20} };

        i = 0;
        PtSetArg( &args[i++], Pt_ARG_TEXT_STRING, "V max (mV):", 0 );
        PtSetArg( &args[i++], Pt_ARG_AREA, &area, 0 );
        PtCreateWidget( PtLabel, NULL, i, args );
        }
        /* 4 */
        {
        PhArea_t        area = { {700, 50}, {50, 20} };

        i = 0;
        PtSetArg( &args[i++], Pt_ARG_TEXT_STRING, "0", 0 );
        PtSetArg( &args[i++], Pt_ARG_AREA, &area, 0 );
        label2 = PtCreateWidget( PtLabel, NULL, i, args );
        }
        /* 5 */
        {
        PhArea_t        area = { {600, 70}, {50, 20} };

        i = 0;
        PtSetArg( &args[i++], Pt_ARG_TEXT_STRING, "V min (mV):", 0 );
        PtSetArg( &args[i++], Pt_ARG_AREA, &area, 0 );
        PtCreateWidget( PtLabel, NULL, i, args );
        }
        /* 6 */
        {
        PhArea_t        area = { {700, 70}, {50, 20} };

        i = 0;
        PtSetArg( &args[i++], Pt_ARG_TEXT_STRING, "0", 0 );
        PtSetArg( &args[i++], Pt_ARG_AREA, &area, 0 );
        label3 = PtCreateWidget( PtLabel, NULL, i, args );
        }
    }

    /* Toggle */
    {
        PhArea_t        area = { {20, 5}, {100, 25} };

        i = 0;
        PtSetArg( &args[i++], Pt_ARG_TEXT_STRING, "Begin draw", 0 );
        PtSetArg( &args[i++], Pt_ARG_AREA, &area, 0 );
        feed_toggle_wgt = PtCreateWidget( PtToggleButton, NULL, i, args );
    }

    /* Mtrend */
    {
        PhArea_t        area = { {20, 30}, {560, 340} };
        PtMTrendAttr_t  graph1_attr;
            graph1_attr.state = Pt_MTREND_STATE_SHOWN;
            graph1_attr.color = Pg_RED;
            graph1_attr.line_thickness = 1;
            graph1_attr.join_type = Pg_MITER_JOIN;
            graph1_attr.min = -5300;
            graph1_attr.max =  5300;
            graph1_attr.draw_f = NULL;

        i = 0;
        PtSetArg( &args[i++], Pt_ARG_MTREND_N_GRAPHS, 1, 0 );
        PtSetArg( &args[i++], Pt_ARG_MTREND_N_SAMPLES, 8 * 350,  0 );
        PtSetArg( &args[i++], Pt_ARG_MTREND_FLAGS, Pt_TRUE, Pt_MTREND_BLIT );
        PtSetArg( &args[i++], Pt_ARG_MTREND_GRAPH_ATTR, &graph1_attr, 0 );
        PtSetArg( &args[i++], Pt_ARG_AREA, &area, 0 );

        mtrend_wgt = PtCreateWidget( PtMTrend, NULL, i, args );
    }

    /* Timer */
    {
        PtCallback_t callback = { timer_func, NULL };

        i = 0;
        PtSetArg( &args[i++], Pt_ARG_TIMER_INITIAL, 100, 0 );
        PtSetArg( &args[i++], Pt_ARG_TIMER_REPEAT, 1, 0 );
        PtSetArg( &args[i++], Pt_CB_TIMER_ACTIVATE, &callback, 0 );

        PtCreateWidget( PtTimer, NULL, i, args );
    }

    init_ioadm_lib( mode, pci_index );

    PtRealizeWidget( window );

    PtMainLoop();

    ioadm_close( &l783m );

    return 0;
}
