/*
 * (c) CBD BC, Russia, Spb.
 *
 * Mail: support@kpda.ru
 *
 * Date: 20/11/2009
 * Dev:  A. Docuchaev
 */


/* main.c */


#include "io-adm-internal.h"


io_adm_t                    *io_adm = NULL;


void parse_options( io_adm_t *io_adm, int argc, char *argv[] )
{
    int                 c;

    while (-1 != (c = getopt(argc, argv, "sd:m:o:p:P:vV"))) {
        switch ( c ) {
            case 's':
                io_adm->multithreading = 0;
                break;
            case 'd':
                io_adm->drivers++;
                io_adm->dll_desc[io_adm->drivers - 1].dll_name = optarg;
                break;
            case 'm':
                io_adm->mem_sz = strtoul( optarg, 0, 0 );
                if ( io_adm->mem_sz < 1024 ) {
                    MSG( STDF, "Warning: Number of bytes in the resource set to default (%u)\n",
                               IO_ADM_ATTR_NB_DEFAULT );
                    io_adm->mem_sz = IO_ADM_ATTR_NB_DEFAULT;
                }
                break;
            case 'o':
                if ( io_adm->drivers > 0 )
                    io_adm->dll_desc[io_adm->drivers - 1].dll_options = optarg;
                else
                    MSG( STDF, "Warning: \"-o\" option before \"-d\"\n" );
                break;
            case 'p':
                io_adm->priority = strtoul( optarg, 0, 0 );
                if ( io_adm->priority < 10 )
                    MSG( STDF, "Warning: priority is too low\n" );
                break;
            case 'P':
                if ( strcmp( optarg, "FIFO" ) == 0 )
                    io_adm->policy = SCHED_FIFO;
                else if ( strcmp( optarg, "RR" ) == 0 )
                    io_adm->policy = SCHED_RR;
                else if ( strcmp( optarg, "SPORADIC" ) == 0 )
                    io_adm->policy = SCHED_SPORADIC;
                else {
                    MSG( STDF, "Warning: Scheduling policy undefined (used to default - RR)\n" );
                    io_adm->policy = IO_ADM_ATTR_POLICY_DEFAULT;
                }
                break;
            case 'v':
                io_adm->verbose = 1;
                break;
            case 'V':
                MSG( STDF, "Version: %1.1f\nDate: %s\n", IO_ADM_VERSION_MAX, IO_ADM_DATE );
                exit( 0 );
            break;
            default:
                MSG( STDF, "Error: Undefined option\n" );
                exit( -1 );
        }
    }
}


int main( int argc, char *argv[] )
{
    char                    dll[255] = "";
    io_adm_drv_init_t       drv_init = NULL;
    struct sched_param      sched_param;
    int                     i = 0,
                            t = 0,
                            c = 0;

    /* Create io-adm manager */
    io_adm = malloc( sizeof(io_adm_t) );
    if ( io_adm == NULL ) {
        MSG( ERRF, "Error: %s(): Can't allocate io_adm_t\n", __FUNCTION__ );
        exit( -1 );
    }
    io_adm->multithreading          = 1;
    io_adm->verbose                 = 0;
    io_adm->drivers                 = 0;    /* Count of drivers */
    io_adm->mem_sz                  = IO_ADM_ATTR_NB_DEFAULT;
    io_adm->priority                = IO_ADM_ATTR_PRIORITY_DEFAULT;
    io_adm->policy                  = IO_ADM_ATTR_POLICY_DEFAULT;
    for ( i = 0; i < IO_ADM_MAX_DEVICE_COUNT; i++ ) {
        io_adm->dll_desc[i].dll_name       = NULL;
        io_adm->dll_desc[i].dll            = NULL;
        io_adm->dll_desc[i].drv_descriptor = NULL;
        io_adm->dll_desc[i].blocked        = 0;
        io_adm->dll_desc[i].loaded         = 0;
    }

    parse_options( io_adm, argc, argv );

    if ( io_adm->verbose ) {
        MSG( STDF, "io-adm v.%1.1f\n", IO_ADM_VERSION_MAX );
        for ( i = 0; i < io_adm->drivers; i++ )
            MSG( STDF, "Driver #%i: %s\n", i, io_adm->dll_desc[i].dll_name );
        MSG( STDF, "Scheduling priority: %u\n", io_adm->priority );
        if ( io_adm->policy == SCHED_FIFO )
            MSG( STDF, "Scheduling policy: FIFO\n" );
        else if ( io_adm->policy == SCHED_RR )
            MSG( STDF, "Scheduling policy: RR\n" );
        else 
            MSG( STDF, "Scheduling policy: SPORADIC\n" );
        MSG( STDF, "Memory: %ub\n", io_adm->mem_sz );
    }

    /* Set priority */
    sched_param.sched_priority = io_adm->priority;
    pthread_setschedparam( pthread_self(), io_adm->policy, &sched_param );

    io_adm_init( io_adm );

    /* Load drivers */
    for ( i = 0; i < io_adm->drivers; i++ ) {

        if ( i >= IO_ADM_MAX_DEVICE_COUNT ) {
            MSG( ERRF, "Error: MAX_DEV_CNT=%i\n", IO_ADM_MAX_DEVICE_COUNT );
            exit( -1 );
        }

        /* Check for driver */
        if ( !io_adm->dll_desc[i].dll_name ) {
            MSG( ERRF, "Error: Driver not specified (see \"-d\" option)\n" );
            exit( -1 );
        }
        sprintf( dll, "%sdevadc-%s.so", IO_ADM_DRV_PATH, io_adm->dll_desc[i].dll_name );
        if ( !(io_adm->dll_desc[i].dll = dlopen( dll, RTLD_NOW )) ) {
            sprintf( dll, "%sdevdac-%s.so", IO_ADM_DRV_PATH, io_adm->dll_desc[i].dll_name );
            if ( !(io_adm->dll_desc[i].dll = dlopen( dll, RTLD_NOW )) ) {
                sprintf( dll, "%sdevdio-%s.so", IO_ADM_DRV_PATH, io_adm->dll_desc[i].dll_name );
                if ( !(io_adm->dll_desc[i].dll = dlopen( dll, RTLD_NOW )) ) {
                    MSG( ERRF, "Error: Driver \"%s\" not found\n", io_adm->dll_desc[i].dll_name );
                    exit( -1 );
                }
            }
        } 

        /* Get descriptor */
        if ( !(drv_init = (io_adm_drv_init_t)dlsym( io_adm->dll_desc[i].dll,
                                                    "io_adm_drv_init" )) ) {
            MSG( ERRF, "Error: Initializer not found\n" );
            exit( -1 );
        }
        if ( !io_adm->multithreading ) {
            if ( !(io_adm->dll_desc[i].drv_descriptor = 
                   (io_adm_drv_descriptor_t *)drv_init( io_adm->dll_desc[i].dll_options )) ) {
                MSG( ERRF, "Error: Initialization failed\n" );
                exit( -1 );
            }
        } else {
            /* Init thread */
            io_adm->dll_desc[i].io_adm_drv_thread = io_adm_drv_thread;
            pthread_attr_init( &io_adm->dll_desc[i].t_attr );
            pthread_attr_setdetachstate( &io_adm->dll_desc[i].t_attr, PTHREAD_CREATE_DETACHED );

            io_adm->drv_index = i;
            io_adm->dll_desc[i].drv_init = drv_init;
            io_adm->dll_desc[i].drv_descriptor = NULL;
            io_adm->t_error = 0;

            if ( pthread_create( &io_adm->dll_desc[i].thread,
                                 &io_adm->dll_desc[i].t_attr,
                                 io_adm->dll_desc[i].io_adm_drv_thread,
                                 io_adm ) != EOK ) {
                MSG( ERRF, "Error: Can't create thread\n" );
                exit( -1 );
            }

            /* Wait for the descriptor to be received */
            while ( io_adm->dll_desc[i].drv_descriptor == NULL ) {
                /* Error occured */
                if ( io_adm->t_error == 1 ) {
                    MSG( ERRF, "Error: Can't initialize driver's thread\n" );
                    exit( -1 );
                }
                /* Sleep 10ms */
                usleep( 10000 );
            }
        }

        /* Init driver */
        io_adm_drv_init( &io_adm->dll_desc[i] );

    }

    /* BUS-scan processing */
    t = io_adm->drivers;
    c = 0;
    for ( i = 0; i < io_adm->drivers; i++ ) {

        int         stop = 0;

        while ( (io_adm->dll_desc[i].drv_descriptor->caps & IO_ADM_DRV_CAPS_BUSSCAN) &&
                (io_adm->dll_desc[i].drv_descriptor->io_adm_drv_busscan_result) &&
                (io_adm->dll_desc[i].drv_descriptor->io_adm_drv_busscan_result()) ) {
            if ( t >= IO_ADM_MAX_DEVICE_COUNT ) {
                MSG( ERRF, "Error: To much bus scanning requests (please increase IO_ADM_MAX_DEVICE_COUNT), scanning stopped\n" );
                stop = 1;
                break;
            }

            /* Check for driver */
            io_adm->dll_desc[t].dll_name = io_adm->dll_desc[i].dll_name;
            io_adm->dll_desc[t].dll_options = io_adm->dll_desc[i].dll_options;
            sprintf( dll, "%sdevadc-%s.so", IO_ADM_DRV_PATH, io_adm->dll_desc[t].dll_name );
            if ( !(io_adm->dll_desc[t].dll = dlopen( dll, RTLD_NOW )) ) {
                sprintf( dll, "%sdevdac-%s.so", IO_ADM_DRV_PATH, io_adm->dll_desc[t].dll_name );
                if ( !(io_adm->dll_desc[t].dll = dlopen( dll, RTLD_NOW )) ) {
                    sprintf( dll, "%sdevdio-%s.so", IO_ADM_DRV_PATH, io_adm->dll_desc[t].dll_name );
                    if ( !(io_adm->dll_desc[t].dll = dlopen( dll, RTLD_NOW )) ) {
                        MSG( ERRF, "Error: Can't load driver \"%s\" for the bus scanning (scanning stopped)\n", io_adm->dll_desc[t].dll_name );
                        stop = 1;
                        break;
                    }
                }
            }

            /* Get descriptor */
            if ( !(drv_init = (io_adm_drv_init_t)dlsym( io_adm->dll_desc[t].dll,
                                                        "io_adm_drv_init" )) ) {
                MSG( ERRF, "Error: Initializer not found (bus scanning stopped)\n" );
                stop = 1;
                break;
            }
            if ( !io_adm->multithreading ) {
                if ( !(io_adm->dll_desc[t].drv_descriptor = 
                       (io_adm_drv_descriptor_t *)drv_init( io_adm->dll_desc[t].dll_options )) ) {
                    MSG( ERRF, "Error: Initialization failed (bus scanning stopped)\n" );
                    stop = 1;
                    break;
                }
            } else {
                /* Init thread */
                io_adm->dll_desc[t].io_adm_drv_thread = io_adm_drv_thread;
                pthread_attr_init( &io_adm->dll_desc[t].t_attr );
                pthread_attr_setdetachstate( &io_adm->dll_desc[t].t_attr, PTHREAD_CREATE_DETACHED );

                io_adm->drv_index = t;
                io_adm->dll_desc[i].drv_init = drv_init;
                io_adm->dll_desc[i].drv_descriptor = NULL;
                io_adm->t_error = 0;

                if ( pthread_create( &io_adm->dll_desc[t].thread,
                                     &io_adm->dll_desc[t].t_attr,
                                     io_adm->dll_desc[t].io_adm_drv_thread,
                                     io_adm ) != EOK ) {
                    MSG( ERRF, "Error: Can't create thread\n" );
                    exit( -1 );
                }

                /* Wait for the descriptor to be received */
                while ( io_adm->dll_desc[t].drv_descriptor == NULL ) {
                    /* Error occured */
                    if ( io_adm->t_error == 1 ) {
                        MSG( ERRF, "Error: Can't initialize driver's thread\n" );
                        exit( -1 );
                    }
                    /* Sleep 10ms */
                    usleep( 10000 );
                }
            }

            /* Init driver */
            io_adm_drv_init( &io_adm->dll_desc[t] );

            t++;
            c++;

        }

        if ( stop )
            break;

    }
    io_adm->drivers += c;

    /* Daemon mode */
    if ( procmgr_daemon( 0, PROCMGR_DAEMON_NOCLOSE |
                            PROCMGR_DAEMON_NODEVNULL ) == -1 ) {
        MSG( ERRF, "Error: Can't do procmgr_daemon()\n" );
        exit( -1 );
    }

    io_adm_event( io_adm );

    return (0);
}
