/*
 * (c) CBD BC, Russia, Spb.
 *
 * Mail: support@kpda.ru
 *
 * Date: 04/10/2010
 * Dev:  A. Docuchaev
 */


/* init.c */


#include "pio.h"


static  char                *piod64_dev_opts [] = {
    "verbose",              /* 0 */
    "idx",                  /* 1 */
    "busscan",              /* 2 */
    NULL
};


/*
 * Called from io_adm_drv_init()
 * Out: EOK - ok
 */
int dev_parse_options( piod64_dev_t *dev, char *options )
{
    char                    *value, *restore, *c;
    int                     opt;
    int                     rc = 0;
    int                     err = EOK;

    if ( !options )
        return EOK;

    restore = NULL;

    while ( options && *options != '\0' ) {

        c = options;
        restore = strchr( options, ',' );
        opt = getsubopt( &options, piod64_dev_opts, &value );

        switch ( opt ) {
            case  0:    /* "verbose" */
                if ( value )
                    dev->verbose = strtoul( value, 0, 0 );
                else
                    dev->verbose = 1;
                break;

            case 1:     /* "idx" */
                if ( value && !(dev->busscan) )
                    dev->descriptor.device_idx = strtoul( value, 0, 0 );
                break;

            case 2:     /* "busscan" */
                dev->busscan = 1;
                break;

            default:
                MSG( STDF, "devdio-piod64.so: Undefined option\n" );
                return EINVAL;
                break;
        }   /* switch */

        if (restore != NULL)
            *restore = ',';

    }   /* while */

    errno = err;
    return (rc);
}


/*
 * io-adm entry point - initialize driver
 *
 * You must ensure that this function is ok to call over and over again in busscan mode
 */
void * io_adm_drv_init( char *options )
{
    piod64_dev_t            *dev = (piod64_dev_t *)malloc( sizeof( piod64_dev_t ) );
    if ( !dev ) {
        fprintf( stderr, "devdio-piod64.so: Can't allocate piod64_dev_t\n" );
        return (NULL);
    }
    memset( dev, 0, sizeof( piod64_dev_t ) );

    /* Init drv-descriptor */
    IO_ADM_DRV_DESCRIPTOR_INIT( dev->descriptor )
    dev->descriptor.io_adm_drv_open           = io_adm_drv_open;
    dev->descriptor.io_adm_drv_close          = io_adm_drv_close;
    dev->descriptor.io_adm_drv_destroy        = io_adm_drv_destroy;
    dev->descriptor.io_adm_drv_write          = io_adm_drv_write;
    dev->descriptor.io_adm_drv_read           = io_adm_drv_read;
    dev->descriptor.io_adm_drv_devctl         = io_adm_drv_devctl;
    dev->descriptor.io_adm_drv_busscan_result = io_adm_drv_busscan_result;
    dev->descriptor.io_adm_drv_devname        = io_adm_drv_devname;
    dev->descriptor.io_adm_drv_devid          = io_adm_drv_devid;
    dev->descriptor.io_adm_drv_devsubid       = io_adm_drv_devsubid;
    dev->descriptor.io_adm_drv_devstat        = io_adm_drv_devstat;
    dev->descriptor.device_idx                = 0;

    /* Parse options */
    if ( dev_parse_options( dev, options ) != EOK ) {
        free( dev );
        return NULL;
    }

    ThreadCtl( _NTO_TCTL_IO, 0 );

    /* Identify & attach device */
    if ( dev->busscan ) {
        if ( piod64_busscan( dev, 0, 0, -1 ) != 0 ) {
            fprintf( stderr, "devdio-piod64.so: PIO-D64 not found\n" );
            return (NULL);
        }

        dev->descriptor.device_idx = dev->idx;
        dev->descriptor.caps       = IO_ADM_DRV_CAPS_BUSSCAN;
    } else {
        if ( piod64_busscan( dev, PCI_VENDOR_ID_ICPDAS, PCI_DEVICE_ID_N0, dev->descriptor.device_idx ) != 0 )
            if ( piod64_busscan( dev, PCI_VENDOR_ID_ICPDAS, PCI_DEVICE_ID_N1, dev->descriptor.device_idx ) != 0 ) {
                fprintf( stderr, "devdio-piod64.so: PIO-D64 not found\n" );
                return (NULL);
            }

        dev->idx                   = dev->descriptor.device_idx;
    }
    dev->descriptor.device_state |= IO_ADM_DRV_STATE_DEV_DETECTED;
    dev->descriptor.device_state |= IO_ADM_DRV_STATE_DEV_MEM_ALLOCATED;

    if ( dev->verbose ) {
        MSG( STDF, "\nICP DAS PIO-D64/PIO-D64U driver started\n" );
        MSG( STDF, "Device found:\n    VID:DID:\t\t%04x:%04x\n", dev->vid, dev->did );
        MSG( STDF, "    PCI-id:\t\t%i\n", dev->idx );
        MSG( STDF, "    Sub-id:\t\t%02x%08x\n", dev->sub_id0, dev->sub_id1 );
        MSG( STDF, "    Name:\t\t%s\n", dev->name );
        MSG( STDF, "    I/O:\t\t%x\n", dev->base );
    }

    /* Init IRQ */
    init_isr( dev );
    /* Enable IRQ in HW */
    dev->iid = InterruptAttach( (int)dev->irq,
                                  irq_handler,
                                  (const void *)dev,
                                  sizeof( piod64_dev_t * ),
                                  _NTO_INTR_FLAGS_TRK_MSK );
    if ( dev->iid < 0 ) {
        MSG( STDF, "devdio-piod64.so: InterruptAttach() failed\n" );
        return NULL;
    }

    /* Init device */
    out8( dev->base + PIO_D64_INT_MASK_CONTROL, 0x0 );      /* Disable IRQs */
    dev->int_mask = in8( dev->base + PIO_D64_INT_MASK_CONTROL ) & 0x0f;
    out8( dev->base + PIO_D64_RESET_CONTROL,    0x0 );      /* Disable I/O operations */
    out8( dev->base + PIO_D64_INT_POLARITY, 1 );            /* Set positive IRQ polarity */
    dev->int_polarity = in8( dev->base + PIO_D64_INT_POLARITY ) & 0x0f;

    dev->descriptor.device_state |= IO_ADM_DRV_STATE_DEV_INITED;
    dev->descriptor.device_state |= IO_ADM_DRV_STATE_READY;

    dev->descriptor.caps |= IO_ADM_DRV_CAPS_DEVNAME |
                            IO_ADM_DRV_CAPS_DEVID |
                            IO_ADM_DRV_CAPS_DEVSUBID |
                            IO_ADM_DRV_CAPS_DEVSTATS;

    return dev;
}

uint8_t io_adm_drv_busscan_result( void )
{
    if ( piod64_busscan_result() == 0 )
        return (1);
    else
        return (0);
}

char * io_adm_drv_devname( void *device )
{
    piod64_dev_t            *dev = (piod64_dev_t *)device;

    if ( dev->attached )
        return dev->name;
    else
        return NULL;
}

uint32_t io_adm_drv_devid( void *device )
{
    piod64_dev_t            *dev = (piod64_dev_t *)device;

    if ( dev->attached )
        return ((dev->vid << 16) | dev->did);

    return (0x0ul);
}

uint64_t io_adm_drv_devsubid( void *device )
{
    piod64_dev_t            *dev = (piod64_dev_t *)device;

    if ( dev->attached )
        return ((uint64_t)dev->sub_id0 << 32 | (uint64_t)dev->sub_id1);

    return (0x0ull);
}

void * io_adm_drv_devstat( void *device )
{
    /* Not realized yet */
    return NULL;
}

int io_adm_drv_open( void *device )
{
    piod64_dev_t            *dev = (piod64_dev_t *)device;

    /* Clear stats */
    IO_ADM_DRV_DESCRIPTOR_CLEAR_STATS( dev->descriptor )

    return (0);
}

void io_adm_drv_close( void *device )
{
    piod64_dev_t            *dev = (piod64_dev_t *)device;

    /* Disable IRQ */
    out8( dev->base + PIO_D64_INT_MASK_CONTROL, 0x0 );
    dev->descriptor.device_state &= ~IO_ADM_DRV_STATE_IRQ_ENABLED;

    /* Disable device */
    out8( dev->base + PIO_D64_RESET_CONTROL,    0x0 );
    dev->descriptor.device_state &= ~IO_ADM_DRV_STATE_DEV_STARTED;

    /* Close device */
    piod64_busscan_detach( dev );
}

void io_adm_drv_destroy( void *device )
{
    piod64_dev_t            *dev = (piod64_dev_t *)device;

    io_adm_drv_close( device );

    /* Detach ISR */
    InterruptDetach( dev->iid );
    dev->iid = 0;

    free( dev );
}
