/*
 * (c) CBD BC, Russia, Spb.
 *
 * Mail: support@kpda.ru
 *
 * Date: 27/11/2009
 * Dev:  A. Docuchaev
 */


/* bios.c */


#include "l783m.h"
#include "bios.h"


/*
 * Load DSP bios
 *
 * In: Device ptr
 *     BIOS file name
 * Out: 0 - ok
 */
int load_dsp_bios( l783m_t *l783m, char *bios_file )
{
    FILE                    *fbios  = NULL;     /* BIOS file */
    uint32_t                sz      = 0;
    uint8_t                 *bios   = NULL;     /* BIOS ptr */
    uint16_t                *usbios = NULL;     /* BIOS (uint16_t *) ptr */
    uint16_t                *ptr    = NULL;     /* Current ptr */
    uint16_t                count   = 0;        /* Count of blocks in uint16_t/uint32_t ubnits */

    if ( strcmp( l783m->bios_file, "" ) == 0 ) {    /* Use preloaded DSP bios */

        sz = L783_bios_sz;
        bios = L783_bios;
        usbios = (uint16_t *)bios;
        if ( l783m->verbose )
            MSG( STDF, "Using default DSP BIOS (date: \"%s\")\n", L783_bios_date );

    } else {

        if ( !(fbios = fopen( bios_file, "rb" )) ) {
            MSG( ERRF, "l783m: Can't open BIOS file \"%s\"\n", bios_file );
            return (-1);
        }

        /* Calculate size */
        fseek( fbios, 0, SEEK_END );
        sz = ftell( fbios );
        rewind( fbios );

        /* Read BIOS */
        bios = (uint8_t *)malloc( sz + 2 );
        if ( bios == NULL ) {
            MSG( ERRF, "l783m: Can't write DSP program memory\n" );
            fclose( fbios );
            return (-1);
        }
        if ( fread( bios, 1, sz, fbios ) != sz ) {
            MSG( ERRF, "l783m: Can't write DSP program memory\n" );
            free( bios );
            fclose( fbios );
            return (-1);
        }
        usbios = (uint16_t *)bios;

    }

    /* Reset  */
    plx_int_enable( l783m, 0 );
    plx_reset( l783m );

    /* Write data smemory */
    ptr = usbios + usbios[0] + 1;   /* Get data memory address */
    count = *ptr++;
    if ( write_dsp_dm( l783m, DSP_DM_START_ADDR, ptr, (uint32_t)count ) != 0 ) {
        MSG( ERRF, "l783m: Can't write DSP data memory\n" );
        free( bios );
        fclose( fbios );
        return (-1);
    }

    /* Write Revision */
    write_dsp_dm_word( l783m, BIOS_VARIABLE_BOARD_REVISION + l783m->board_info.bi.base_dsp_dm_offset, (unsigned short)'C' );

    /* Write programm memory */
    ptr = &usbios[3];               /* Get programm memnory address */
    count = (uint16_t)(usbios[0] - 2);
    if ( write_dsp_pm( l783m, 0x1, (uint32_t *)ptr, count/2 ) != 0 ) {
        MSG( ERRF, "l783m: Can't write DSP program memory\n" );
        free( bios );
        fclose( fbios );
        return (-1);
    }
    sz = *((uint32_t *)&usbios[1]);
    if ( write_dsp_pm( l783m, 0x0, &sz, 1 ) != 0 ) {
        MSG( ERRF, "l783m: Can't write DSP program memory\n" );
        free( bios );
        fclose( fbios );
        return (-1);
    }

    plx_int_enable( l783m, 1 );

    /* Test DSP BIOS */
    if ( (sz = (uint32_t)test_bios( l783m )) != 0 ) {
        MSG( ERRF, "l783m: DSP BIOS test failed (test #%i)\n", (int)sz );
        free( bios );
        fclose( fbios );
        return (-1);
    }

    l783m->bios_loaded = 1;

    if ( strcmp( l783m->bios_file, "" ) != 0 ) {
        free( bios );
        fclose( fbios );
    }
    return (0);
}


/*
 * Test DSP bios
 *
 * In: Device ptr
 * Out: 0 - ok
 *      N - Failed test number
 */
int test_bios( l783m_t *l783m )
{
    int                     x = 0;

    /* Test 1 */
    x = read_dsp_dm_word( l783m, BIOS_VARIABLE_TMODE1 + l783m->board_info.bi.base_dsp_dm_offset );
    if( x != 0x5555 ) {
        if( l783m->verbose )
            MSG( ERRF, "l783m: Test #1: value: 0x%x\n", x );
        return (1);
    }

    /* Test 2 */
    x = read_dsp_dm_word( l783m, BIOS_VARIABLE_TMODE2 + l783m->board_info.bi.base_dsp_dm_offset );
    if( x != 0xAAAA ) {
        if( l783m->verbose )
            MSG( ERRF, "l783m: Test #2: value: 0x%x\n", x );
        return (2);
    }

    /* Test 3 */
    write_dsp_dm_word( l783m, BIOS_VARIABLE_TEST_LOAD + l783m->board_info.bi.base_dsp_dm_offset, 0x77BB );
    for( x = 64; x > 0; x-- ) {
        if ( read_dsp_dm_word( l783m, BIOS_VARIABLE_READY + l783m->board_info.bi.base_dsp_dm_offset ) != 1 )
            break;
        usleep( 20 );   /* 20 microseconds */
    }
    if ( (x = read_dsp_dm_word( l783m, BIOS_VARIABLE_READY + l783m->board_info.bi.base_dsp_dm_offset )) != 1 ) {
        if( l783m->verbose )
            MSG( ERRF, "l783m: BIOS is not ready\n" );
        return (3);
    }

    /* Test 4 */
    if ( dsp_bios_command( l783m, BIOS_COMMAND_TEST, 0 ) < 0 )
        if( l783m->verbose )
            MSG( ERRF, "l783m: TEST command error\n" );
    x = read_dsp_dm_word( l783m, BIOS_VARIABLE_TEST_LOAD + l783m->board_info.bi.base_dsp_dm_offset );
    if ( x != 0xAA55 ) {
        if( l783m->verbose )
            MSG( ERRF, "l783m: TEST_LOAD = %04x\n", x );
        return (4);
    }

    bios_set_dsp_type( l783m );

    return (0);
}


/*
 * Set DSP type to BIOS
 *
 * In: Device ptr
 */
void bios_set_dsp_type( l783m_t *l783m )
{
    char                    *dsp[]            = { "2184", "2185", "2186", NULL };
    int                     adc_fifo_max_sz[] = { 0x800, 0x3800, 0x1800, 0 };
    int                     adc_index[]       = { 2, 0, 1, 0 };
    int                     i                 = 0;
    int                     x                 = 0;

    write_dsp_dm_word( l783m, BIOS_VARIABLE_READY + l783m->board_info.bi.base_dsp_dm_offset, 0x0 );

    while ( dsp[i] ) {
        if ( strncmp( l783m->board_info.eeprom.dsp_type, dsp[i], 4 ) == 0 )
            break;
        i++;
    }

    l783m->adc_fifo_max_sz = adc_fifo_max_sz[i];
    write_dsp_dm_word( l783m, BIOS_VARIABLE_DSP_TYPE + l783m->board_info.bi.base_dsp_dm_offset, i );
    dsp_bios_command( l783m, BIOS_COMMAND_SET_DSP_TYPE, 0 );
    if ( i != 0 )
        l783m->board_info.bi.base_dsp_dm_offset = 0x3000;

    write_dsp_dm_word( l783m, BIOS_VARIABLE_DSP_TYPE + l783m->board_info.bi.base_dsp_dm_offset, i );

    for( x = 0; x < 100; x++ ) {
        if ( read_dsp_dm_word( l783m, BIOS_VARIABLE_READY + l783m->board_info.bi.base_dsp_dm_offset ) != 1 )
            break;
        usleep( 20 );   /* 20 microseconds */
    }

    write_dsp_dm_word( l783m, BIOS_VARIABLE_ADC_FIFO_BASE_INDEX + l783m->board_info.bi.base_dsp_dm_offset, adc_index[i] );
    dsp_bios_command( l783m, BIOS_COMMAND_ADC_FIFO_CFG, 0 );
}
