//	Rhapsody		: 7.1 
//	Component		: oxfFiles 
//	Configuration 	: generic
//	Model Element	: OMSelfLinkedMemoryAllocator
//!	File name		: $Source: R:/StmOO/Master/cg/LangCpp/oxf/rcs/OMSelfLinkedMemoryAllocator.h $
//!	File version	: $Revision: 1.18 $
//
//!	Date changed	: $Date: 2007/04/06 07:07:49 $
//!	Last change by	: $Author: ilgiga $
//
//	(c) Copyright Telelogic 2004, 2007
//



#ifndef OMSelfLinkedMemoryAllocator_H 

#define OMSelfLinkedMemoryAllocator_H 

#include "IOxfMemoryAllocator.h"
#include "OMMemoryManagerWrapper.h"
#include "omprotected.h"

//----------------------------------------------------------------------------
// OMSelfLinkedMemoryAllocator.h                                                                  
//----------------------------------------------------------------------------

//## package Design::oxf::Services::MemoryManagement::Pools 


// Memory pool implementation
//## class OMSelfLinkedMemoryAllocator 
template <class T, int INITNUM> class RP_FRAMEWORK_DLL OMSelfLinkedMemoryAllocator : public IOxfMemoryAllocator {
public :
    //#[ ignore 
    // Additional memory allocation callback
    typedef omtypename T* (*AllocationCallback)(int);
    //#]

////    Constructors and destructors    ////
public :
    
    // construct the allocator, specify whether it is protected
    // and how much additional memory should be allocated if the initial pool is exhausted
    // Argument int incrementNum : 
    // The number of additional instances to allocate when the initial pool is empty.
    // Argument bool isProtected : 
    // Enable/disable memory pool protection from race between threads
    //## operation OMSelfLinkedMemoryAllocator(int,bool) 
    OMSelfLinkedMemoryAllocator(int incrementNum, bool isProtected) : allocator_(0) ,headOfFreeList_(0) ,incrementNum_(
    incrementNum) ,isProtected_(isProtected) ,memoryPoolIsEmptyFlag_(true) ,myGuard_(false) {
        //#[ operation OMSelfLinkedMemoryAllocator(int,bool) 
        if (isProtected_) {
        	myGuard_.initializeMutex();
        }
        
        lock();
        {
        	T* newBlock = reinterpret_cast<T*>(&initialBlock_);
        	initiatePool(newBlock,INITNUM);
        }
        unlock();
        //#]
    }
    
    
    // Cleanup
    //## operation ~OMSelfLinkedMemoryAllocator() 
    ~OMSelfLinkedMemoryAllocator() {
        //#[ operation ~OMSelfLinkedMemoryAllocator() 
        allocator_ = 0;
        headOfFreeList_ = 0;
        //#]
    }
    


////    Operations    ////
public :
    
    // allocate memory pool big enough to hold numOfInstances instances of type T
    // Argument int numOfInstances : 
    // The number of instances to allocate memory for
    //## operation allocPool(int) 
    inline T* allocPool(int numOfInstances) {
        //#[ operation allocPool(int) 
        // allocate a big block of memory
        T* newBlock = NULL;
        
        // notice, that according to the ANSI standard allocating 0 bytes 
        // return a valid pointer and NOT 0
        if (numOfInstances > 0) {
        #ifdef USE_DYNAMIC_MEMORY_ALLOCATION		
        	// the initial pool was used so we will have to allocate another chunk using ::new
        	// (still, we benefit by minimizing fragmentation and  lowering the overhead of many new-s)
        	newBlock = reinterpret_cast<T*>(OMMemoryManagerWrapper::getMemory(sizeof(T) * numOfInstances));
        #endif // USE_DYNAMIC_MEMORY_ALLOCATION
        }
        return newBlock;
        //#]
    }
    
    
    // Get the memory allocation callback function
    //## operation getAllocator_() const 
    inline AllocationCallback getAllocator_() const {
        //#[ operation getAllocator_() const 
        return allocator_;
        //#]
    }
    
    // get a memory block of the specified size
    // Argument size_t size : 
    // The memory to allocate (in bytes)
    //## operation getMemory(size_t) 
    inline virtual void * getMemory(size_t size) {
        //#[ operation getMemory(size_t) 
        if ((size != 0) && (size != sizeof(T))) {
        #ifdef USE_DYNAMIC_MEMORY_ALLOCATION		
        	return OMMemoryManagerWrapper::getMemory(size);
        #else
        	return NULL;
        #endif // USE_DYNAMIC_MEMORY_ALLOCATION
        }
        
        T* p = NULL;
        lock();
        {
        	p = headOfFreeList_;
        
        	// if p is not valid, allocate more space
        	if (p == NULL) {
        		// hook for the user to catch pool exhaustion allocation of more memory
        		// by setting a breakpoint in T::OMMemoryPoolIsEmpty()
        		if (memoryPoolIsEmptyFlag_) {
        			T::OMMemoryPoolIsEmpty();	// notification
        		}
        		T* newBlock = NULL;
        		if (allocator_ != NULL) {
        			newBlock = allocator_(incrementNum_);
        		}
        		else {
        			newBlock = allocPool(incrementNum_);
        		}
        		// connect the pool
        		initiatePool(newBlock,incrementNum_);
        		p = headOfFreeList_;
        	}
        	// if p is valid, just move the list head to the
        	// next element in the free list
        	if (p != NULL) { 
        		headOfFreeList_ = p->OMMemoryPoolNextChunk;
        	}
        }
        unlock();
        
        return p;
        //#]
    }
    
    
    // initiate the bookkeeping for the allocated pool
    // Argument T* const newBlock : 
    // The new memory block
    // Argument int numOfInstances : 
    // The number of instances in the memory block
    //## operation initiatePool(T* const,int) 
    inline int initiatePool(T* const newBlock, int numOfInstances) {
        //#[ operation initiatePool(T* const,int) 
        // make sure we got one
        if ((newBlock == NULL) || (numOfInstances == 0)) {
        	headOfFreeList_ = NULL;
        	return 0;
        }
        
        // link the objects together
        for (int i = 0; i < numOfInstances - 1; i++) {
        	newBlock[i].OMMemoryPoolNextChunk = &newBlock[i+1];
        }
        
        // terminate the linked list with a null pointer
        newBlock[numOfInstances-1].OMMemoryPoolNextChunk = NULL;
        
        // set headOfFreeList_ to the first element
        headOfFreeList_ = &newBlock[0];          
        return 1;
        //#]
    }
    
    
    // return a memory of object of the specified size
    // Argument void * object : 
    // The object to delete
    // Argument size_t size : 
    // The size of the object (in bytes)
    //## operation returnMemory(void *,size_t) 
    inline virtual void returnMemory(void * object, size_t size) {
        //#[ operation returnMemory(void *,size_t) 
        if (object == 0) {
        	return;
        }
        
        // send objects of "wrong" size to ::op delete
        if ((size != 0) && (size != sizeof(T))) {
          OMMemoryManagerWrapper::Delete(object, size);
          return;
        }
        
        lock();
        {
        	T* carcass = reinterpret_cast<T*>(object);
        	carcass->OMMemoryPoolNextChunk = headOfFreeList_;
        	headOfFreeList_ = carcass;
        }
        unlock();
        //#]
    }
    
    
    // Set the memory pool additional memory allocation callback
    // Argument AllocationCallback p_allocator_ : 
    // The callback
    //## operation setAllocator(AllocationCallback) 
    inline void setAllocator(AllocationCallback p_allocator_) {
        //#[ operation setAllocator(AllocationCallback) 
        allocator_ = p_allocator_;
        //#]
    }
    

private :
    
    // lock critical section
    //## operation lock() const 
    inline void lock() const {
        //#[ operation lock() const 
        if (isProtected_) {
        	myGuard_.lock();
        }
        //#]
    }
    
    
    // unlock critical section
    //## operation unlock() const 
    inline void unlock() const {
        //#[ operation unlock() const 
        if (isProtected_) {
        	myGuard_.unlock();
        }
        //#]
    }
    


////    Additional operations    ////
public :
    
    //## auto_generated 
    inline int getIncrementNum() const {
        return incrementNum_;
    }
    
    
    //## auto_generated 
    inline void setIncrementNum(int p_incrementNum_) {
        incrementNum_ = p_incrementNum_;
    }
    
    
    //## auto_generated 
    inline bool getIsProtected() const {
        return isProtected_;
    }
    
    
    //## auto_generated 
    inline void setIsProtected(bool p_isProtected_) {
        isProtected_ = p_isProtected_;
    }
    
    
    //## auto_generated 
    inline void callMemoryPoolIsEmpty(bool p_memoryPoolIsEmptyFlag_) {
        memoryPoolIsEmptyFlag_ = p_memoryPoolIsEmptyFlag_;
    }
    


////    Attributes    ////
private :
    
    // callback function to override the memory allocation scheme
    AllocationCallback allocator_;		//## attribute allocator_ 
    
    // head of the free list
    T* headOfFreeList_;		//## attribute headOfFreeList_ 
    
    // how much (in instance number) to increment if initial pool is exhausted
    int incrementNum_;		//## attribute incrementNum_ 
  
    // initial pool

/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////

	//unsigned char initialBlock_[(OMRAW_MEMORY_ALIGNMENT-1)+(INITNUM*sizeof(T))];		//## attribute initialBlock_ 
	//unsigned char initialBlock_[(OMRAW_MEMORY_ALIGNMENT-1)+(INITNUM*T::SIZE_OF_T_TYPE())];		//## attribute initialBlock_ 
	#if INITNUM == 0
	unsigned char initialBlock_[(OMRAW_MEMORY_ALIGNMENT-1)];		//## attribute initialBlock_ 
	#else
	unsigned char initialBlock_[(OMRAW_MEMORY_ALIGNMENT-1)+(INITNUM*sizeof(T))];		//## attribute initialBlock_ 
	#endif

/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////

    // activate the mutex
    bool isProtected_;		//## attribute isProtected_ 
    
    // when the flag is false, memoryPoolIsEmpty() will not be called
    bool memoryPoolIsEmptyFlag_;		//## attribute memoryPoolIsEmptyFlag_ 
    

////    Relations and components    ////
private :
    
    // The pool guard
    OMProtected myGuard_;		//## classInstance myGuard_ 
    


};
//## class OMSelfLinkedMemoryAllocator 





#endif  
//
//! Log: $Log: OMSelfLinkedMemoryAllocator.h $
//! Log: Revision 1.18  2007/04/06 07:07:49  ilgiga
//! Log: bug fix 98191
//


