#ifndef omsdata_H
#define omsdata_H "$Id: omsdata.h 1.23 2007/03/11 12:34:37 ilgiga Exp $"

//
//	file name   :	$Source: R:/StmOO/Master/cg/LangCpp/omcom/rcs/omsdata.h $
//	file version:	$Revision: 1.23 $
//
//	purpose: define the OMSData - Serialized Data class	
//			This is the "base class" for all messages between
//			AOM and TOM
//
//	author(s):	Yachin Pnueli
//	date started:	21.5.96
//	date changed:	$Date: 2007/03/11 12:34:37 $
//	last change by:	$Author: ilgiga $
//
//	(c) Copyright Telelogic 1995, 2007
//

#include <oxf/rp_framework_dll_definition.h>
#include <oxf/rawtypes.h>
#include <omcom/omnote.h>

//
//	OMSxxxx/AOMSxxx/TOMSxxxx
//	are the classes implementing the protocol through which data is 
//	transmitted from executable to animator/tracer and back.
//
//	OMSData which handles generic data is the lowest level of this protocol
//	OMSData items are composed of a code + data.
//	The Possibilities are:
//	1. instanceCode then pointer (5 bytes).
//		used also for some integers known to be 4 bytes or less.
//  2. stringCode then size (1 byte) then data - string less than 255 long
//		or the first segment of longer astrings.
//	3. continueCode then size (1 bytes) then data - A continuation of a 
//		previous string.
//	4. serializedDataCode - item is a serialized data
//  5. emptyCode then nothing indicates an empty string.
//	6. maxCode - Maximal code "used" for actual codes
//		Codes larger than this indicate various "code" items -
//		small single chars, AOMNotifies, AOMCommands and other small "enum's"
//
//	Data OutSide the animation framework (i.e. User Data) should be sent
//	as either:
// (AOMInstance *) - for classes derived from AOXFInstance
//			via the x2item template
//	or 
//	(char *)/OMString - for other classes.
//	Note: Classes having a "<<" operation can use the x2String template.
//		see	(om2str.h)


// OMSPosition - the 'index' into an OMSData item
// Should be replaced either by an internal iterator (readable OMSDATA
// or by an open standard iterator.
typedef int OMSPosition;


RP_FRAMEWORK_DLLV(extern "C") void sendError(const char *);
// the error routine of omCom
// In OMOMATE this is implemented as tomSendError
// In OMAPPLICATION this is implemented as aomSendError

// The value returned if no "good" character can be read
const unsigned char badCharValue = 255;

const int OMSAllLevels = (-1);

// Error handling routines
RP_ANIM_DLL void lostSyncError(OMString& errMsg, OMSPosition position);


class RP_ANIM_DLL RP_FRAMEWORK_DLL OMSData {
	enum OMSCode {	// Codes for the items which can be stored in an OMSData
		pointerCode  = 1,	// Following item is a pointer
		stringCode	 = 2,	// Following item is a string
		continueCode = 3,	// Following item is the continuation of a previous string
		emptyCode	 = 4,	// A special code for empty strings (no following item)
		sdataCode	 = 5,	// Following item is an sdataCode
		timeunitCode = 6,	// Following item is a time unit
		maxCode		 = 7	// All "Code" codes must be >maxCode
	};
	const char * omscode2String(OMSCode); // Converts OMSCode to string (char *)

	OMString data;	// Where data is actually held.

	// Consistancy checking routines (print errors if found)
	OMBoolean badPosition(OMSPosition);
	OMBoolean badPosition(OMString&, OMSPosition);
	OMBoolean badPositionOrCode(OMSPosition, OMSCode);

protected:
	void addItemAtPosition(const timeUnit, OMSPosition);
	void addItemAtPosition(const void *, OMSPosition);
	void addItemAtPosition(const int i, OMSPosition p) { 
		addItemAtPosition((void *)i,p);
	}
	void addEmptyString() { data+=(unsigned char)emptyCode; }
	void addItem(const char * item, int size);
public:
	virtual ~OMSData() { }
	OMSData() { }
	// Constructors which already place the first item inside
	OMSData(const char * item)			{ addItem(item); }
	OMSData(const void * item)			{ addItem(item); }
	OMSData(const int item)				{ addItem(item); }
	OMSData(const OMString * item)		{ addItem(item); }
	OMSData(const unsigned char item)	{ addItem(item); }
	OMSData(const OMSData& item)		{ addItem(item); }
	OMSData(const timeUnit item)		{ addItem(item); }

	// Constructors which already place the message headers inside
	OMSData(const void * destination, OMNotify what)
	{
		addItem(destination); addItem(what);
	}
	
	OMSData(const void * destination, OMNotify what, const char * item)
	{
		addItem(destination); addItem(what); addItem(item);
	}
	
	OMSData(const void * destination, OMNotify what, const void * item)
	{
		addItem(destination); addItem(what); addItem(item);
	}
	
	OMSData(const void * destination, OMNotify what, const int item)
	{
		addItem(destination); addItem(what); addItem(item);
	}
	
	OMSData(const void * destination, OMNotify what, const OMString * item)
	{
		addItem(destination); addItem(what); addItem(item);
	}
	
	OMSData(const void * destination, OMNotify what, const unsigned char item)
	{
		addItem(destination); addItem(what); addItem(item);
	}
	
	OMSData(const void * destination, OMNotify what, const bool item)
	{
		addItem(destination); addItem(what); addItem(item);
	}
	
	OMSData(const void * destination, OMNotify what, const OMSData& item)
	{
		addItem(destination); addItem(what); addItem(item);
	}

	// Methods to add items to an AOMSData 
	void addCode(const unsigned char item)
	{
		if (item <= 255-maxCode)
			data += (unsigned char)(item+maxCode);
		else
			sendError("Trying to addCode for an invalid item");
	}	

	virtual void addItem(const unsigned char item)
	{
		addCode(item);
	}
	
	virtual void addItem(const bool item)
	{
		const int i = item;
		addItem(i);
	}
	
	virtual void addItem(const OMNotify item)
	{
		addItem((unsigned char)item);
	
	}
	
	virtual void addItem(const OMSData& item) 
	{
		data+= (char)sdataCode; 
		OMString tmp = item.data;
		addItem(tmp.GetBuffer(0),tmp.GetLength());
	}

	virtual void addItem(const timeUnit item);
	virtual void addItem(const void * item);
	
	virtual void addItem(const int item) 
	{ 
		addItem((void *)item); 
	}
	
	virtual void addItem(const char * item) 
	{
		if (item==NULL)
			addEmptyString();
		else
			addItem(item,strlen(item));
	}
	
	virtual void addItem(const OMString * item) 
	{
		if (item==NULL)
			addEmptyString();
		else {
			OMString tmp = (*item);
			addItem(tmp.GetBuffer(0),tmp.GetLength());
		}
	}
	
	virtual void addItem(const OMString& item) 
	{ 
		OMString tmp = item;
		addItem(tmp.GetBuffer(0),tmp.GetLength());
	}

	// concat data with no code (unsafe)
	virtual void concatData(const OMSData& omsData)
	{
		data += omsData.data;
	}

	// Methods to check the type of the next item
	OMBoolean isString(OMSPosition position)	{ 
		return data[position]==stringCode || data[position]==emptyCode;	}
	OMBoolean isPointer(OMSPosition position)	{ return data[position]==pointerCode;}
	OMBoolean isInt(OMSPosition position)	{ return data[position]==pointerCode;}
	OMBoolean isTimeUnit(OMSPosition position)	{ return data[position] == timeunitCode;}
	OMBoolean isSData(OMSPosition position)	{ 
		return data[position]==sdataCode || data[position]==emptyCode;	}
	OMBoolean isCode(OMSPosition position)	{ return data[position]>=maxCode;		}
	OMBoolean isEmpty(OMSPosition position)	{ return position>=data.GetLength(); }
	OMBoolean isNotEmpty(OMSPosition position)	{ return position<data.GetLength();	}
	OMBoolean isStringOrPointer(OMString& s, OMSPosition position);

	// Unsafe methods to get the next item -- 
	// use "getX()" only if caller checked "isX()"
	// else use safeGetX()
	char*		getChar(OMSPosition& position);

	OMString	getString(OMSPosition& position);

	timeUnit	getTimeUnit(OMSPosition& position);

	void*		getPointer(OMSPosition& position);
	int			getInt(OMSPosition& position) {
		return (int)getPointer(position);
	}

	unsigned char getCode(OMSPosition& position)	{

// disable warning about conversion from 'int' to 'unsigned char' - this is safe
#ifdef OM_WIN32_COMPILER_DISABLE_WARNING_4244
#pragma warning ( push )
#pragma warning ( disable : 4244 )
#endif // OM_WIN32_COMPILER_DISABLE_WARNING_4244
		
		return (unsigned char)data[position++]-maxCode; 

#ifdef OM_WIN32_COMPILER_DISABLE_WARNING_4244
#pragma warning ( pop )
#endif // OM_WIN32_COMPILER_DISABLE_WARNING_4244

	}

	OMNotify getOMNotify(OMSPosition& position)	{ 
		return (OMNotify)getCode(position); 
	}

	OMSData * getOMSData(OMSPosition& position);


	// Safe methods to get the next item -- 
	// Check that next item is of the appropriate type
	// If so return value, else return "bad value"
	char* safeGetChar(OMSPosition& position) {
		if(badPositionOrCode(position,stringCode)) return (char *)("");
		else return getChar(position);
	}

	OMString safeGetString(OMSPosition& position){
		if(badPositionOrCode(position,stringCode)) return "";
		else return getString(position);
	}

	timeUnit safeGetTimeUnit(OMSPosition& position) {
		if(badPositionOrCode(position,timeunitCode)) return 0;
		else return getTimeUnit(position);
	}

	void*	safeGetPointer(OMSPosition& position) {
		if(badPositionOrCode(position,pointerCode)) return NULL;
		else return getPointer(position);
	}

	int		safeGetInt(OMSPosition& position) {
		if(badPositionOrCode(position,pointerCode)) return 0;
		else return getInt(position);
	}

	unsigned char safeGetCode(OMSPosition& position)	{ 
		if(badPositionOrCode(position,maxCode)) return (badCharValue);
		else return getCode(position); 
	}

	OMNotify	safeGetOMNotify(OMSPosition& position)	{ 
		if(badPositionOrCode(position,maxCode)) return (OMNotify)(badCharValue);
		else return (OMNotify)getCode(position); 
	}

	OMSData *	safeGetOMSData(OMSPosition& position){
		if(badPositionOrCode(position,sdataCode)) return new OMSData("");
		else return getOMSData(position);
	}

	OMSPosition getFirst() { return 0; } // Return the beginning of an AOMSData
	int getLength() const { return data.GetLength(); }

	// Coomparison operators
	OMBoolean operator==(OMSData& other) { return data==other.data; }
	OMBoolean operator!=(OMSData& other) { return data!=other.data; }

	// Needed for the "sending through sockets"
	const char * getRawData() { return data.GetBuffer(0); }
	static OMSData * string2OMSData(const char* data) {
		OMSData * msg = new OMSData();
		msg->data = data;
		return msg;
	}

	// To avoid psosX86 Warnings regarding no match with OMSListData
	// Feel free to add implementations if necessary
	virtual void addItem(const OMHandle*, const OMString* ) { }
	virtual void addItem(const OMHandle*, const char* ) { }
	virtual void addItem(const OMHandle*, const void* ) { }

};

// 


class RP_ANIM_DLL RP_FRAMEWORK_DLL OMSListData : public OMSData
{

public:
	OMSListData(int theCount = countUnknown) : count(0) { 
		// set count and position
		if (theCount==countUnknown) {
			// Remember the position where "count" should be placed
			position = getLength();
			 // Place there a dummy to advance the position
			OMSData::addItem(0);
		} else {
			 // Remember there is no need to remember the count position
			position = positionUnneeded;
			// Store the actual "count" in place
			OMSData::addItem(theCount); 
		}
	}

	virtual void addItem(const char * value)	{ count++; OMSData::addItem(value); }
	virtual void addItem(const int value)	{ count++; OMSData::addItem(value); }
	virtual void addItem(const OMString* value) { count++; OMSData::addItem(value); }
	virtual void addItem(const OMString& value) { count++; OMSData::addItem(value); }
	virtual void addItem(const unsigned long value) { count++; OMSData::addItem(value); }
	virtual void addItem(const OMNotify value) { count++; OMSData::addItem(value); }
	virtual void addItem(const unsigned char value) { count++; OMSData::addItem(value); }
	virtual void addItem(const bool value) { count++; OMSData::addItem(value); }
	virtual void addItem(const void * value) { count++; OMSData::addItem(value);	 }
	// For double value
	virtual void addItem(const OMHandle* name, const OMString* value)
	{
		count++;
		OMSData::addItem(name);
		OMSData::addItem(value);
	}

	virtual void addItem(const OMHandle* name, const char * value)
	{
		count++; 
		OMSData::addItem(name);
		OMSData::addItem(value);
	}

	virtual void addItem(const OMHandle* name, const void* value)
	{
		count++; 
		OMSData::addItem(name);
		OMSData::addItem(value);
	}

	virtual void addItemAndItsProxy(const void* item)
	{
		count++; 
		OMSData::addItem(item);
		OMSData::addItem(item);
	}

	virtual void addItem(const OMSData& msg)
	{
		count++; 
		OMSData::addItem(msg);
	}
	
	int end()
	{
		// Write the count if necessary
		if (position!=positionUnneeded)
			addItemAtPosition(count, position);
		return count;
	}

	int getCount()
	{
		return count;
	}

private:
	int count;
	int position;

	static const int countUnknown; // Param./Attr. count of an item is unknown
	static const int positionUnneeded; 
};


//
// $Log: omsdata.h $
// Revision 1.23  2007/03/11 12:34:37  ilgiga
// Change copyright comment
// Revision 1.22  2007/03/01 16:43:33  ilgiga
// Telelogic instead of i-Logix
// Revision 1.21  2004/06/27 15:42:41  amos
// move to model-based oxf in RiC++
// Revision 1.20  2004/05/19 08:21:50  amos
// Revision 1.17  2002/07/07 11:42:56  amos
// framework cleanup from RTOS specific code - back to r41 main branch
// Revision 1.16.1.2  2002/07/07 11:42:56  amos
// replace adaptor specific #ifdef with generic statements
// Revision 1.16.1.1  2002/01/01 08:51:37  amos
// Duplicate revision
// Revision 1.15  2001/06/21 10:34:50  Eldad
// Solaris warnings.
// Revision 1.14.1.1  2001/06/21 08:45:24  builder
// Revision 1.14  2001/05/17 10:38:44  Eldad
// psosX86 Warnings.
// Revision 1.13  2001/05/16 16:01:19  Eldad
// psosppc fix.
// Revision 1.12  2001/04/02 07:50:33  Eldad
// Added OMSListData.
// Made all addItem methods virtual.
// Revision 1.11  2001/01/04 18:49:08  amos
// merge 1.8.1.1 (COM DLL) into r31
// Revision 1.10  2000/12/25 07:15:22  amos
// fix use of #ifdef WIN32
// Revision 1.9  2000/12/21 17:36:26  amos
// move to error level 4
// Revision 1.8.1.1  2000/12/27 09:02:02  amos
// Add support for framework as DLL - for support in Rhapsody C++ COM
// Revision 1.8  2000/07/19 13:24:19  beery
// Comply with Solaris strict checking of assign "" to char*
// Revision 1.7  2000/07/12 06:50:00  amos
// the main change related to modify char* to const char*.
// there are some other changes related to assignment of static member function as extern "C" functions.
// Revision 1.6  2000/01/31 09:45:57  amos
// back to main branch
// Revision 1.5.1.2  2000/01/26 14:26:37  amos
// add timeUnit to OMSData - all the standard interfaces
// Revision 1.5.1.1  1999/11/08 14:01:52  amos
// Duplicate revision
// Revision 1.4  1999/06/02 15:16:01  amos
// support connection of C++ TOM to RiC
// Revision 1.3  1999/05/11 09:54:35  ofer
// Windows CE changes
// no iostream support and unicode ONLY ( no exceptions)
// Revision 1.2  1998/08/02 15:08:59  beery
// changing boolean->OMBoolean
// Revision 1.1  1996/11/24 12:37:22  yachin
// Initial revision
//

#endif
