/*
 * (c) CBD BC, Russia, Spb.
 *
 * Mail: support@kpda.ru
 *
 * Date: 23/11/2009
 * Dev:  A. Docuchaev
 */


/* io.c */


#include "l783m.h"


/* Defines */

#define DEVCTL_RET( cond, devctl_status )       \
    if ( cond ) {                               \
        msg->o.ret_val = (devctl_status);       \
        break;                                  \
    }


/* Funcs */

/*
 * _IO_READ handler
 *
 * In: Device ptr
 *     Message ptr (Message size >= 4)
 *     Size to read
 * Out: n - count of readed bytes
 *      0 - error
 */
int io_adm_drv_read( void *device, void *msg, uint32_t size )
{
    l783m_t                 *l783m   = (l783m_t *)device;
    l783m_msg_t             *msg2    = (l783m_msg_t *)msg;
    int                     read_sz  = 0;

    /* DSP BIOS is damaged */
    if ( !l783m->bios_loaded )
        return (0);

    /* Buffer size not enough to store shared-memory offset */
    if ( size < 4 )
        return (0);

    /* ADC Disabled */
    if ( !l783m->adc_start )
        return (0);

    if ( l783m->ir_fixed ) {

        InterruptWait( 0, NULL );

        read_sz = size;     /* Size client ready to read */
        l783m->i_rbuf.avail_sz = l783m->i_rbuf.fragmentation_sz;  /* Buffer has */
        if ( read_sz > l783m->i_rbuf.avail_sz )
            read_sz = l783m->i_rbuf.avail_sz;
        if ( read_sz > l783m->i_rbuf.sz )
            read_sz = l783m->i_rbuf.sz;
        if ( read_sz > 0 ) {
            /* Send offset */
            *((uint32_t *)(&msg2->data)) = 0;

            l783m->i_rbuf.read_idx = 0;
        }

    } else {

        /* No available data - wait for it */
        while ( l783m->i_rbuf.write_idx == l783m->i_rbuf.read_idx )
            InterruptWait( 0, NULL );

        read_sz = size;     /* Size client ready to read */
        l783m->i_rbuf.avail_sz = rbuf_avail_sz( &l783m->i_rbuf ); /* Buffer has */
        if ( read_sz > l783m->i_rbuf.avail_sz )
            read_sz = l783m->i_rbuf.avail_sz;
        if ( l783m->i_rbuf.read_idx + read_sz > l783m->i_rbuf.sz )
            read_sz = l783m->i_rbuf.sz - l783m->i_rbuf.read_idx;
        if ( read_sz > 0 ) {
            /* Send offset */
            *((uint32_t *)(&msg2->data)) = l783m->i_rbuf.read_idx;

            /* Move read ptr */
            l783m->i_rbuf.read_idx = (l783m->i_rbuf.read_idx + read_sz) % l783m->i_rbuf.sz;
        }

    }

    l783m->descriptor.adc_data_readed += read_sz;
    return (read_sz);
}

/*
 * _IO_WRITE handler
 *
 * In: Device ptr
 *     Message ptr
 *     Message size
 * Out: -1 - error
 */
int io_adm_drv_write( void *device, void *msg, uint32_t size )
{
    /*l783m_t                 *l783m = (l783m_t *)device;*/

    /* Not permitted */

    return (-1);
}

/*
 * _IO_DEVCTL handler
 *
 * In: Device ptr
 *     io_devctl_t message ptr
 *     Buffer ptr
 * Out: 0 - ok
 */
int io_adm_drv_devctl( void *device, io_devctl_t *msg, void *data )
{
    l783m_t                 *l783m = (l783m_t *)device;

    switch ( msg->i.dcmd ) {

        case DCMD_IOADM_CHANNELS: {
            int             status = 0;

            DEVCTL_RET( *(int *)data > ADC_MAX_CHANNELS, 1 )
            DEVCTL_RET( *(int *)data <= 0, 4 )
            status = dsp_adc_enable( l783m, 0, 0 );
            DEVCTL_RET( status != 0, 5 )

            l783m->channels = *(int *)data;
            write_dsp_dm_word( l783m, BIOS_VARIABLE_CONTROL_HEIGHT +
                                      l783m->board_info.bi.base_dsp_dm_offset, l783m->channels );
            status = dsp_bios_command( l783m, BIOS_COMMAND_LOAD_CONTROL_TABLE, 0 );
            DEVCTL_RET( status != 0, 6 )

            //status = dsp_adc_enable( l783m, 1, 0 );
            //DEVCTL_RET( status != 0, 5 )

            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_SET_CHANNEL: {
            l783m_msg_channel_t *ch_msg = (l783m_msg_channel_t *)data;

            if ( !(ch_msg->mode & L783M_CHANNEL_PRESET) ) {
                ch_msg->x  = ch_msg->input & 0x3F;
                ch_msg->x |= (ch_msg->gain & 3) << 6;
                if (ch_msg->mode & L783M_CHANNEL_SINGLE_ENDED)
                    ch_msg->x |= 0x20;
                if (ch_msg->mode & L783M_CHANNEL_CALIBRATION)
                    ch_msg->x |= 0x10;
            }

            DEVCTL_RET( ch_msg->index >= ADC_MAX_CHANNELS, 1 )
            DEVCTL_RET( ch_msg->index < 0,                 1 )

            write_dsp_dm_word( l783m, BIOS_VARIABLE_CONTROL_TABLE + ch_msg->index +
                                      l783m->board_info.bi.base_dsp_dm_offset, ch_msg->x );

            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_SET_IN_BUFF: {
            l783m_buff_info_t   *bi_msg = (l783m_buff_info_t *)data;
            int                 status  = 0;

            DEVCTL_RET( bi_msg->buf_sz < 0,   1 )
            DEVCTL_RET( bi_msg->frag_sz <= 0, 1 )

            free_rbuf( &l783m->i_rbuf );

            if ( bi_msg->buf_sz == 0 )      /* Set to maximum */
                bi_msg->buf_sz = RBUF_SZ - (RBUF_SZ % bi_msg->frag_sz);

            init_rbuf( l783m, &l783m->i_rbuf, bi_msg->buf_sz, bi_msg->frag_sz );

            status = dsp_adc_set_buffer( l783m, 0, bi_msg->frag_sz );
            DEVCTL_RET( status != 0, 2 )

            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_SET_FRATE: {
            int                 status = 0;
            int                 cr     = 0;

            plx_int_enable( l783m, 0 );

            status = dsp_adc_enable( l783m, 0, 0 );
            DEVCTL_RET( status != 0, 1 )

            read_dsp_dm_word( l783m, BIOS_VARIABLE_CONTROL_HEIGHT + l783m->board_info.bi.base_dsp_dm_offset );

            DEVCTL_RET( *(int *)data <= 0, 2 )

            l783m->rate = *(int *)data;

            status = dsp_adc_set_timing( l783m, &l783m->rate, &cr );
            DEVCTL_RET( status != 0, 3 )

            //status = dsp_adc_enable( l783m, 1, 0 );
            //DEVCTL_RET( status != 0, 4 )

            status = dsp_adc_set_buffer( l783m, 0 /* Max value */, l783m->i_rbuf.fragmentation_sz );
            DEVCTL_RET( status != 0, 5 )

            //status =  dsp_adc_int_enable( l783m, 1, 0 );
            //DEVCTL_RET( status != 0, 6 )

            plx_int_enable( l783m, 1 );

            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_SET_TIMING: {
            l783m_msg_timing_params_t   *tp    = (l783m_msg_timing_params_t *)data;
            int                         status = 0;

            status = dsp_adc_set_timing( l783m, &tp->frame_rate, &tp->channel_rate );
            DEVCTL_RET( status != 0, 1 )

            l783m->rate = tp->frame_rate;

            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_SET_SYNC: {
            l783m_msg_sync_t    *s     = (l783m_msg_sync_t *)data;
            int                 status = 0;
            int                 tmp    = -1;
            int                 tmp2  = -1;

            if ( s->mode != L783M_SYNC_MODE_NONE ) {
                plx_int_enable( l783m, 0 );

                //status = dsp_adc_int_enable( l783m, 1, 0 );
                //DEVCTL_RET( status != 0, 1 )
            }

            if ( s->mode & L783M_SYNC_MODE_DIGITAL ) {
                if ( s->mode & L783M_SYNC_MODE_FRAME )
                    tmp = 0;  /* Digital Sync + Per-Frame Sync */
                else
                    tmp = 1;  /* Digital Sync */
            }
            else
                if ( s->mode & L783M_SYNC_MODE_ANALOG )
                    tmp = 2;  /* Analog Sync */

            write_dsp_dm_word( l783m, BIOS_VARIABLE_SYNC_TYPE +
                                      l783m->board_info.bi.base_dsp_dm_offset, tmp );
            write_dsp_dm_word( l783m, BIOS_VARIABLE_SYNC_CHANNEL +
                                      l783m->board_info.bi.base_dsp_dm_offset, s->channel );
            write_dsp_dm_word( l783m, BIOS_VARIABLE_SYNC_LEVEL +
                                      l783m->board_info.bi.base_dsp_dm_offset, s->threshold );

            switch ( s->mode & 0x0f ) {
                case L783M_SYNC_MODE_HIGH:
                    tmp = 0;
                    tmp2 = 0;
                    break;
                case L783M_SYNC_MODE_LOW:
                    tmp = 0;
                    tmp2 = 1;
                    break;
                case L783M_SYNC_MODE_RISE:
                    tmp = 1;
                    tmp2 = 0;
                    break;
                case L783M_SYNC_MODE_FALL:
                    tmp = 1;
                    tmp2 = 1;
                    break;
            }

            write_dsp_dm_word( l783m, BIOS_VARIABLE_SYNC_EDGE +
                                      l783m->board_info.bi.base_dsp_dm_offset, tmp );
            write_dsp_dm_word( l783m, BIOS_VARIABLE_SYNC_MODE +
                                      l783m->board_info.bi.base_dsp_dm_offset, tmp2 );

            status = dsp_bios_command( l783m, BIOS_COMMAND_SYNC_CONFIG, 0 );
            DEVCTL_RET( status != 0, 2 )

            if ( s->mode != L783M_SYNC_MODE_NONE ) {
                plx_int_enable( l783m, 1 );
                l783m->adc_start = 1;       /* Start ADC reading */
            }

            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_START: {
            int             status = 0;

            DEVCTL_RET( l783m->adc_start == 1, 1 )      /* Already started */

            l783m->adc_start = 1;

            status = dsp_bios_command( l783m, BIOS_COMMAND_LOAD_CONTROL_TABLE, 0 );
            DEVCTL_RET( status != 0, 2 )

            status = dsp_adc_enable( l783m, 1, 0 );
            DEVCTL_RET( status != 0, 3 )

            l783m->i_rbuf.write_idx = l783m->i_rbuf.read_idx = 0;

            status = dsp_adc_int_enable( l783m, 1, 0 );    /* Enable ADC IRQ */
            DEVCTL_RET( status != 0, 4 )

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_STOP: {
            int             status = 0;

            l783m->adc_start = 0;

            status = dsp_adc_enable( l783m, 0, 0 );
            DEVCTL_RET( status != 0, 2 )

            status = dsp_adc_int_enable( l783m, 0, 0 );    /* Disable ADC IRQ */
            DEVCTL_RET( status != 0, 1 )

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_BUFF_MODE: {
            msg->o.ret_val = LIBMODE_SHARED_MEMORY | LIBMODE_BUFPERM_R;
            break;
        }

        case DCMD_IOADM_GET_SHMOBJECT: {
            char            *name = (char *)data;

            strcpy( name, l783m->i_rbuf.shm_name );

            msg->o.nbytes  = 40;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_SHMOBJECT_SZ: {
            msg->o.ret_val = l783m->i_rbuf.sz;
            break;
        }

        case DCMD_IOADM_GET_SHMOBJECT_RD: {
            int                     sz = 0;

            sz = io_adm_drv_read( l783m, data, l783m->i_rbuf.fragmentation_sz );

            msg->o.nbytes  = sizeof( int );
            msg->o.ret_val = sz;
            break;
        }

        case DCMD_IOADM_GET_SAMPLE: {
            l783m_msg_channel_t     *ch_msg = (l783m_msg_channel_t *)data;
            int                     status  = 0;
            signed short int        val     = 0;

            DEVCTL_RET( ch_msg->index >= ADC_MAX_CHANNELS, 1 )
            DEVCTL_RET( ch_msg->index < 0,                 1 )

            if ( !(ch_msg->mode & L783M_CHANNEL_PRESET) ) {
                ch_msg->x  = ch_msg->input & 0x3F;
                ch_msg->x |= (ch_msg->gain & 3) << 6;
                if (ch_msg->mode & L783M_CHANNEL_SINGLE_ENDED)
                    ch_msg->x |= 0x20;
                if (ch_msg->mode & L783M_CHANNEL_CALIBRATION)
                    ch_msg->x |= 0x10;
            }

            write_dsp_dm_word( l783m, BIOS_VARIABLE_ADC_CHANNEL +
                                      l783m->board_info.bi.base_dsp_dm_offset, ch_msg->x );
            write_dsp_dm_word( l783m, BIOS_VARIABLE_ADC_SAMPLE +
                                      l783m->board_info.bi.base_dsp_dm_offset, 0x8000 );

            status = dsp_bios_command( l783m, BIOS_COMMAND_ADC_SAMPLE, 0 );
            DEVCTL_RET( status != 0, 2 )

            val = (signed short int)(uint16_t)read_dsp_dm_word( l783m, BIOS_VARIABLE_ADC_SAMPLE + 
                                                                l783m->board_info.bi.base_dsp_dm_offset );
            *(signed short int *)data = val;

            msg->o.nbytes  = sizeof( l783m_msg_channel_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_SET_CALIBRATION_MUL: {
            int                     status = 0;
            uint16_t                calibration_muls[4];
            uint16_t                scale_muls[4];

            status = eeprom_read_data( l783m, 20, calibration_muls, 4 );
            DEVCTL_RET( status != 0, 1 )

            status = eeprom_read_data( l783m, 24, scale_muls, 4 );
            DEVCTL_RET( status != 0, 2 )

            status = write_dsp_dm( l783m, BIOS_VARIABLE_ZERO + l783m->board_info.bi.base_dsp_dm_offset, calibration_muls, 4 );
            DEVCTL_RET( status != 0, 3 )

            status = write_dsp_dm( l783m, BIOS_VARIABLE_SCALE + l783m->board_info.bi.base_dsp_dm_offset, scale_muls, 4 );
            DEVCTL_RET( status != 0, 4 )

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_ENABLE_CALIBRATION: {
            int                     enable = *(int *)data;

            DEVCTL_RET( enable > 1 || enable < 0, 1 )

            write_dsp_dm_word( l783m, BIOS_VARIABLE_CORRECTION_ENABLE +
                                      l783m->board_info.bi.base_dsp_dm_offset, enable );

            msg->o.nbytes  = 0;
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_SAMPLE_RANGES: {
            l783m_msg_sample_ranges_t   *sr = (l783m_msg_sample_ranges_t *)data;

            sr->sample_val_min = -2048;
            sr->sample_val_max =  2048;

            switch ( sr-> gain ) {

                case 0:
                    sr->u_min          = -5000000;
                    sr->u_max          =  5000000;
                    break;

                case 1:
                    sr->u_min          = -2500000;
                    sr->u_max          =  2500000;
                    break;

                case 2:
                    sr->u_min          = -1250000;
                    sr->u_max          =  1250000;
                    break;

                case 3:
                    sr->u_min          = -625000;
                    sr->u_max          =  625000;
                    break;

                default:
                    msg->o.ret_val = 1;     /* Error status */
                    msg->o.nbytes  = 0;
                    return (0);

            }

            msg->o.nbytes  = sizeof( l783m_msg_sample_ranges_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_INT_TIME: {
            l783m->adc_int_time = (double)(l783m->adc_int_iv_cc) /
                                          (SYSPAGE_ENTRY(qtime)->cycles_per_sec);

            *(double *)data = l783m->adc_int_time;

            msg->o.nbytes  = sizeof( double );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_RD_TIME: {
            l783m->adc_read_time = (double)(l783m->adc_read_iv_cc) /
                                          (SYSPAGE_ENTRY(qtime)->cycles_per_sec);

            *(double *)data = l783m->adc_read_time;

            msg->o.nbytes  = sizeof( double );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_STATE: {
            *(uint32_t *)data = l783m->descriptor.device_state;

            msg->o.nbytes  = sizeof( uint32_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_RECEIVED_DATA_SZ: {
            *(uint64_t *)data = l783m->descriptor.adc_data_received;

            msg->o.nbytes  = sizeof( uint64_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_READED_DATA_SZ: {
            *(uint64_t *)data = l783m->descriptor.adc_data_readed;

            msg->o.nbytes  = sizeof( uint64_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_AVAIL_DATA_SZ: {
            *(uint32_t *)data = 0;

            if ( l783m->ir_fixed ) {
                if ( l783m->descriptor.device_state & IO_ADM_DRV_STATE_IRQ_ENABLED )
                    *(uint32_t *)data = l783m->i_rbuf.fragmentation_sz;
                else
                    *(uint32_t *)data = 0;
            } else
                if ( l783m->descriptor.device_state & IO_ADM_DRV_STATE_IRQ_ENABLED )
                    *(uint32_t *)data = rbuf_avail_sz( &l783m->i_rbuf );
                else
                    *(uint32_t *)data = 0;

            msg->o.nbytes  = sizeof( uint32_t );
            msg->o.ret_val = 0;
            break;
        }

        case DCMD_IOADM_GET_LAST_RECV_TIME: {
            *(uint64_t *)data = l783m->adc_int_ed_cc;

            msg->o.nbytes  = sizeof( uint64_t );
            msg->o.ret_val = 0;
            break;
        }

        default:
            msg->o.ret_val = 0xff;  /* Error status */
            msg->o.nbytes  = 0;

    }

    return (0);
}
