/*
 * (c) CBD BC, Russia, Spb.
 *
 * Mail: support@kpda.ru
 *
 * Date: 22/11/2009
 * Dev:  A. Docuchaev
 */


/* init.c */


#include "l783m.h"


static  char                *l783m_opts [] = {
    "verbose",              /* 0 */
    "idx",                  /* 1 */
    "bios",                 /* 2 */
    "ir_fixed",             /* 3 */
    NULL
};


/*
 * Called from io_adm_drv_init()
 * Out: EOK - ok
 */
int l783m_parse_options( l783m_t *l783m, char *options )
{
    char                    *value, *restore, *c;
    int                     opt;
    int                     rc = 0;
    int                     err = EOK;

    if ( !options )
        return EOK;

    restore = NULL;

    while ( options && *options != '\0' ) {

        c = options;
        restore = strchr( options, ',' );
        opt = getsubopt( &options, l783m_opts, &value );

        switch ( opt ) {
            case  0:    /* "verbose" */
                if ( value )
                    l783m->verbose = strtoul( value, 0, 0 );
                else
                    l783m->verbose = 1;
                break;

            case 1:     /* "idx" */
                if ( value )
                    l783m->descriptor.device_idx = strtoul( value, 0, 0 );
                break;

            case 2:     /* "bios" */
                strcpy( l783m->bios_file, value );
                break;

            case 3:     /* "ir_fixed" */
                l783m->ir_fixed = 1;
                break;

            default:
                MSG( STDF, "l783m: Undefined option\n" );
                return EINVAL;
                break;
        }   /* switch */

        if (restore != NULL)
            *restore = ',';

    }   /* while */

    errno = err;
    return (rc);
}


void * io_adm_drv_init( char *options )
{
    l783m_t                 *l783m = (l783m_t *)malloc( sizeof( l783m_t ) );
    if ( !l783m ) {
        fprintf( stderr, "l783m: Can't allocate l783m_t\n" );
        return (NULL);
    }
    l783m->verbose                 = 0;
    l783m->ir_fixed                = 0;
    l783m->idma_addr               = -1;
    l783m->bios_loaded             = 0;
    l783m->adc_int_st_cc           = 0;
    l783m->adc_int_ed_cc           = 0;
    l783m->adc_int_iv_cc           = 0;
    l783m->adc_int_time            = 0;
    strcpy( l783m->bios_file, "" );

    /* Init drv-descriptor */
    IO_ADM_DRV_DESCRIPTOR_INIT( l783m->descriptor )
    l783m->descriptor.io_adm_drv_open    = io_adm_drv_open;
    l783m->descriptor.io_adm_drv_close   = io_adm_drv_close;
    l783m->descriptor.io_adm_drv_destroy = io_adm_drv_destroy;
    l783m->descriptor.io_adm_drv_write   = io_adm_drv_write;
    l783m->descriptor.io_adm_drv_read    = io_adm_drv_read;
    l783m->descriptor.io_adm_drv_devctl  = io_adm_drv_devctl;
    l783m->descriptor.device_idx         = 0;

    /* Parse options */
    if ( l783m_parse_options( l783m, options ) != EOK )
        return NULL;

    if ( l783m->verbose )
        MSG( STDF, "\nDriver for L-CARD L-783M (rev.C) ADC started\n" );

    ThreadCtl( _NTO_TCTL_IO, 0 );

    /* Find L783m device */
    if ( scan_pci( l783m ) != 0 )
        return NULL;
    l783m->descriptor.device_state |= IO_ADM_DRV_STATE_DEV_DETECTED;

    /* Init IRQ */
    init_isr();
    plx_int_enable( l783m, 0 );
    l783m->iid = InterruptAttach( (int)l783m->board_info.InterruptLine,
                                  irq_handler,
                                  (const void *)l783m,
                                  sizeof( l783m_t * ),
                                  _NTO_INTR_FLAGS_TRK_MSK );
    if ( l783m->iid < 0 ) {
        MSG( STDF, "l783m: InterruptAttach() failed\n" );
        return NULL;
    }

    /* Init rbuf */
    if ( init_rbuf( l783m, &l783m->i_rbuf, RBUF_SZ, RBUF_FRAGMENTATION_SZ ) != 0 )
        return NULL;
    if ( l783m->verbose > 2 )
        MSG( STDF, "irBuffer: size=0x%X/0x%X addr=0x%X\n", l783m->i_rbuf.sz,
                l783m->i_rbuf.fragmentation_sz, (uint32_t)l783m->i_rbuf.data );
    l783m->descriptor.device_state |= IO_ADM_DRV_STATE_DEV_MEM_ALLOCATED;

    io_adm_drv_open( (void *)l783m );

    /* Load BIOS */
    if ( load_dsp_bios( l783m, l783m->bios_file ) != 0 ) {
        MSG( ERRF, "l783m: Can't load BIOS\n" );
        return NULL;
    } else
        if ( l783m->verbose )
            if ( strcmp( l783m->bios_file, "" ) != 0 )
                MSG( STDF, "DSP BIOS loaded (\"%s\")\n", l783m->bios_file );
            else
                MSG( STDF, "Default DSP BIOS loaded\n" );
    l783m->descriptor.device_state |= IO_ADM_DRV_STATE_DEV_INITED;

    l783m->descriptor.device_state |= IO_ADM_DRV_STATE_READY;

    return l783m;
}

int io_adm_drv_open( void *device )
{
    l783m_t                 *l783m = (l783m_t *)device;
    int                     data   = 0;

    plx_int_enable( l783m, 0 );

    l783m->i_rbuf.write_idx = 0;
    l783m->i_rbuf.read_idx  = 0;
    l783m->rate             = 1000;
    l783m->adc_start        = 0;

    IO_ADM_DRV_DESCRIPTOR_CLEAR_STATS( l783m->descriptor )

    if ( l783m->bios_loaded ) {
        l783m->channels = 1;

        data = read_dsp_dm_word( l783m, BIOS_VARIABLE_ADC_FIFO_BASE +
                                        l783m->board_info.bi.base_dsp_dm_offset );
        if ( data )
            write_dsp_dm_word( l783m, BIOS_VARIABLE_ADC_FIFO_PTR +
                                      l783m->board_info.bi.base_dsp_dm_offset, data );
        l783m->i_rbuf_index = data;

        data = l783m->i_rbuf.fragmentation_sz / sizeof( uint16_t );
        write_dsp_dm_word( l783m, BIOS_VARIABLE_IRQ_STEP +
                                  l783m->board_info.bi.base_dsp_dm_offset, data );
    }

    plx_int_enable( l783m, 1 );

    out16( l783m->board_info.bi.base + DSP_RESET_ADDRESS, 0 );

    return (0);
}

void io_adm_drv_close( void *device )
{
    l783m_t                 *l783m = (l783m_t *)device;

    l783m->adc_start = 0;

    plx_int_enable( l783m, 0 );

    dsp_adc_int_enable( l783m, 0, 0 );
    dsp_adc_enable( l783m, 0, 0 );

    l783m->i_rbuf_index = -1;

    plx_int_enable( l783m, 1 );
}

void io_adm_drv_destroy( void *device )
{
    l783m_t                 *l783m = (l783m_t *)device;

    if ( l783m->pci_attach_handle != -1 )
        pci_detach( l783m->pci_attach_handle );

    io_adm_drv_close( device );

    plx_int_enable( l783m, 0 );

    free_rbuf( &l783m->i_rbuf );
    free( l783m );
}
