/*
 * (c) CBD BC, Russia, Spb.
 *
 * Mail: support@kpda.ru
 *
 * Date: 22/11/2009
 * Dev:  A. Docuchaev
 */


/* l783m.h */


#include <io-adm-drv.h>
#include <io-adm-msg.h>
#include <ioadm.h>
#include <L-CARD.h>
#include <l783m_msg.h>

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <malloc.h>
#include <unistd.h>
#include <devctl.h>
#include <fcntl.h>
#include <sys/neutrino.h>
#include <sys/syspage.h>
#include <sys/mman.h>
#include <hw/pci.h>
#include <hw/inout.h>


#ifndef _CBD_BC_IO_ADM_L783M_H_
#define _CBD_BC_IO_ADM_L783M_H_


/* Defs */
#define RBUF_SZ                         0x20000
#define RBUF_FRAGMENTATION_SZ           0x200

#define ADC_MAX_CHANNELS                128
#define ADC_MAX_FREQ                    1000000

#define EEPROM_UDELAY                   10                      /* microseconds */

#define DSP_DM_START_ADDR               0x2000
#define DSP_INTR                        0x04
#define DSP_RESET_ADDRESS               0x06

#define MSG                             fprintf
#define ERRF                            stderr
#define STDF                            stdout


/* Types */

typedef struct lcard_board_info {

    /* PCI Configuration Space */
    uint16_t                    DeviceID;
    uint16_t                    VendorID;
    uint16_t                    StatusReg;
    uint16_t                    CommandReg;
    uint32_t                    ClassCode;
    uint8_t                     RevisionID;
    uint8_t                     BIST;
    uint8_t                     HeaderType;
    uint8_t                     LatencyTimer;
    uint8_t                     CacheLineSize;
    uint32_t                    BAR0;
    uint32_t                    BAR1;
    uint32_t                    BAR2;
    uint32_t                    BAR3;
    uint32_t                    BAR4;
    uint32_t                    BAR5;
    uint32_t                    CardbusCISpointer;
    uint16_t                    SubsystemID;
    uint16_t                    SubsystemVendorID;
    uint32_t                    ExpansionROMBaseAddress;
    uint8_t                     CapabilitiesPointer;
    uint8_t                     Max_Lat;
    uint8_t                     Min_Gnt;
    uint8_t                     InterruptPin;
    uint8_t                     InterruptLine;

    /* Board Info */
    struct BOARD_INFO {
        uint32_t                ioplx;              /* PLX control registers */             /* ConfigRegsBaseAddress */
        uint32_t                cntrl;              /* PLX CNTRL control register */
        uint32_t                eecs;               /* EEPROM Chip Select register */
        uint32_t                base;               /* I/O base address */                  /* IO_BaseAddress */
        uint32_t                base_dsp_dm_offset; /* Data area offset in DSP DM */
        void                    *array;             /* Mapped IDMA array */                 /* HighMemorySpaceBaseAddress */
    }                           bi;

    /* EEPROM Data */
    struct EEPROM_INFO {
        char                    serial[9];
        char                    name[5];
        char                    revision;
        char                    dsp_type[5];
        long                    freq;
        unsigned short          is_dac;
        unsigned short          _res[7];
    }                           eeprom;

} __attribute__ ((__packed__)) lcard_board_info_t;

typedef struct i_rbuf {

    void                        *data;
    char                        shm_name[40];
    int                         shm_fd;
    uint32_t                    sz;
    volatile uint32_t           write_idx;  /* Write ptr */
    volatile uint32_t           read_idx;   /* Read ptr */
    uint32_t                    fragmentation_sz;
    uint32_t                    avail_sz;   /* Available data in the i_rbuf */

} __attribute__ ((__packed__)) i_rbuf_t;

typedef struct l783m {

    io_adm_drv_descriptor_t     descriptor;     /* MUST BE FIRST */

    /* Misc */
    int                         verbose;
    int                         iid;                /* Interrupt ID */

    /* PCI */
    int                         pci_attach_handle;
    void                        *pci_handle;

    /* LCARD */
    char                        bios_file[100];
    lcard_board_info_t          board_info;
    uint8_t                     bios_loaded;

    /* I/O */
    volatile int                ir_fixed;           /* In-buffer pointers are fixed */
    volatile int                adc_start;          /* Start flags */
    int                         adc_fifo_max_sz;    /* Max ADC FIFO size */
    int                         channels;           /* Number of active ADC channels */
    int                         rate;               /* ADC rate */
    volatile int                idma_addr;          /* idma_addr (see irq_handler()) */
    i_rbuf_t                    i_rbuf;             /* In-buffer */
    volatile int                i_rbuf_index;       /* In-buffer index */
    volatile uint64_t           adc_int_st_cc;      /* ADC interrupt start clock-cycles */
    volatile uint64_t           adc_int_ed_cc;      /* ADC interrupt end clock-cycles */
    volatile uint64_t           adc_int_iv_cc;      /* ADC interrupt interval clock-cycles */
    double                      adc_int_time;       /* ADC interrupt time (in sec) */
    volatile uint64_t           adc_read_iv_cc;     /* ADC read interval clock-cycles */
    double                      adc_read_time;      /* ADC read time (in sec) */

} __attribute__ ((__packed__)) l783m_t;


/* Funcs */

/* init.c */
int l783m_parse_options( l783m_t *l783m, char *options );
void * io_adm_drv_init( char *options );
char * io_adm_drv_devname( void *device );
uint32_t io_adm_drv_devid( void *device );
void * io_adm_drv_devstat( void *device );
int io_adm_drv_open( void *device );
void io_adm_drv_close( void *device );
void io_adm_drv_destroy( void *device );

/* io.c */
int io_adm_drv_read( void *device, void *msg, uint32_t size );
int io_adm_drv_write( void *device, void *msg, uint32_t size );
int io_adm_drv_devctl( void *device, io_devctl_t *msg, void *data );

/* l-card.c */
void plx_reset( l783m_t *l783m );
void plx_int_enable( l783m_t *l783m, int enable );
int dsp_adc_int_enable( l783m_t *l783m, int enable, int unblock );
int dsp_adc_enable( l783m_t *l783m, int enable, int unblock );
int scan_pci( l783m_t *l783m );
int write_dsp_pm( l783m_t *l783m, uint32_t addr, uint32_t *data,  uint32_t size );
int write_dsp_dm( l783m_t *l783m, uint32_t addr, uint16_t *data, uint32_t size );
int read_dsp_dm( l783m_t *l783m, uint32_t addr, uint16_t *data, uint32_t size );
int read_dsp_dm_word( l783m_t *l783m, uint32_t addr );
void write_dsp_dm_word( l783m_t *l783m, uint32_t addr, uint16_t val );
int dsp_bios_command( l783m_t *l783m, uint16_t command, uint8_t unblock );
int dsp_adc_set_buffer( l783m_t *l783m, uint32_t size, uint32_t fsize );
int dsp_adc_set_timing( l783m_t *l783m, int *ff, int *cf );

/* eeprom.c */
int eeprom_read_data( l783m_t *l783m, uint32_t addr, uint16_t *buf, uint32_t size );

/* event.c */
void init_isr( void );
const struct sigevent * irq_handler( void *area, int id );

/* bios.c */
int load_dsp_bios( l783m_t *l783m, char *bios_file );
int test_bios( l783m_t *l783m );
void bios_set_dsp_type( l783m_t *l783m );


/* -------- */

/*
 * Fast copy data
 *
 * In: destination ptr
 *     source ptr
 *     size
 */
static inline void copy_32( uint8_t *dst, uint8_t *src, uint32_t len )
{
    int u0, u1, u2;
    __asm__ __volatile__ (
        "cld                \n"
        "rep   movsd        \n"
        "testb $2,%b4       \n"
        "je    1f           \n"
        "movsw              \n"
        "1:                 \n"
        "testb $1,%b4       \n"
        "je    2f           \n"
        "movsb              \n"
        "2:                   "
        : "=&c" (u0), "=&D" (u1), "=&S" (u2)
        : "0" ((unsigned)(len)/4), "q" (len), "1" (dst),"2" (src)
        : "memory" 
    );
}

/*
 * Initialize r-buf
 *
 * In: rbuf ptr
 *     size
 *     fragmentation size
 * Out: 0 - ok
 */
static inline int init_rbuf( l783m_t *l783m, i_rbuf_t *rbuf, uint32_t size, uint32_t fsize )
{
    sprintf( rbuf->shm_name, "/l783m_buf%i", l783m->descriptor.device_idx );

    if ( (rbuf->shm_fd = shm_open( rbuf->shm_name, O_RDWR | O_CREAT, 0777 )) == -1 )
        return (-1);

    /* Set size */
    if ( ftruncate( rbuf->shm_fd, size ) == -1 ) {
        close( rbuf->shm_fd );
        shm_unlink( rbuf->shm_name );
        return (-1);
    }

    if ( (rbuf->data = (void *)mmap( 0, size, PROT_READ | PROT_WRITE,
                                     MAP_SHARED, rbuf->shm_fd, 0 )) == MAP_FAILED ) {
        close( rbuf->shm_fd );
        shm_unlink( rbuf->shm_name );
        return (-1);
    }

    rbuf->sz        = size;
    rbuf->write_idx = 0;
    rbuf->read_idx  = 0;
    rbuf->fragmentation_sz = fsize;

    return (0);
}

/*
 * Free r-buf
 *
 * In: rbuf ptr
 *     size
 *     fragmentation size
 * Out: 0 - ok
 */
static inline int free_rbuf( i_rbuf_t *rbuf )
{
    if ( rbuf->data == NULL )
        return (-1);

    munmap( rbuf->data, rbuf->sz );
    close( rbuf->shm_fd );
    shm_unlink( rbuf->shm_name );

    rbuf->data      = NULL;
    rbuf->sz        = 0;
    rbuf->write_idx = 0;
    rbuf->read_idx  = 0;
    rbuf->fragmentation_sz = 0;

    return (0);
}

/*
 * Get r-buf available data size (in bytes)
 */
static inline int rbuf_avail_sz( i_rbuf_t *rbuf )
{
    int                     tmp;

    tmp = rbuf->write_idx - rbuf->read_idx;

    while ( tmp < 0 )
        tmp += rbuf->sz;

    return tmp;
}


#endif  /* _CBD_BC_IO_ADM_L783M_H_ */
