/*
 * (c) CBD BC, Russia, Spb.
 *
 * Mail: support@kpda.ru
 *
 * Date: 24/11/2009
 * Dev:  A. Docuchaev
 */


/* l-card.c */


#include "l783m.h"


/*
 * Reset PLX
 */
void plx_reset( l783m_t *l783m )
{
    int x;

    /* Reset */
    x = in32( l783m->board_info.bi.cntrl );
    x |= PLX_REG_CNTRL_RESET;
    out32( l783m->board_info.bi.cntrl, x );

    usleep( EEPROM_UDELAY );

    /* un-reset */
    x &= ~PLX_REG_CNTRL_RESET;
    out32( l783m->board_info.bi.cntrl, x );

    usleep( EEPROM_UDELAY );

    return;
}


/*
 * Enable PLX PCI9030 Interrupt
 *
 * In: Device ptr
 *     Enable (1) / Disable (0) interrupt
 */
void plx_int_enable( l783m_t *l783m, int enable )
{
    int x;

    x = PLX_REG_INTCSR_LI1P | PLX_REG_INTCSR_LI2P;
    if ( enable )
        x |= PLX_REG_INTCSR_PIE | PLX_REG_INTCSR_LI1E;
    out32( l783m->board_info.bi.ioplx + PLX_REG_INTCSR, x );

    return;
}


/*
 * Enable ADC PC Interrupt
 *
 * In: Device ptr
 *     Enable (1) / Disable (0) interrupt
 *     Unblock thread immediately (0 is returned)
 * Out: 0  - ok
 *      -1 - error
 */
int dsp_adc_int_enable( l783m_t *l783m, int enable, int unblock )
{
    write_dsp_dm_word( l783m, BIOS_VARIABLE_ENABLE_IRQ_VALUE + l783m->board_info.bi.base_dsp_dm_offset, enable );

    if ( unblock )
        dsp_bios_command( l783m, BIOS_COMMAND_ENABLE_IRQ, 1 );
    else
        if ( dsp_bios_command( l783m, BIOS_COMMAND_ENABLE_IRQ, 0 ) != 0 )
            return (-1);

    if ( enable )
        l783m->descriptor.device_state |= IO_ADM_DRV_STATE_IRQ_ENABLED;
    else
        l783m->descriptor.device_state &= ~IO_ADM_DRV_STATE_IRQ_ENABLED;
    return (0);
}


/*
 * Enable ADC (DSP command)
 *
 * In: Device ptr
 *     Enable (1) / Disable (0)
 *     Unblock thread immediately (0 is returned)
 * Out: 0  - ok
 *      -1 - error
 */
int dsp_adc_enable( l783m_t *l783m, int enable, int unblock )
{
    write_dsp_dm_word( l783m, BIOS_VARIABLE_ADC_ENABLE + l783m->board_info.bi.base_dsp_dm_offset, enable );

    if ( unblock )
        dsp_bios_command( l783m, BIOS_COMMAND_ENABLE_ADC, 1 );
    else
        if ( dsp_bios_command( l783m, BIOS_COMMAND_ENABLE_ADC, 0 ) != 0 )
            return (-1);

    if ( enable )
        l783m->descriptor.device_state |= IO_ADM_DRV_STATE_DEV_STARTED;
    else
        l783m->descriptor.device_state &= ~IO_ADM_DRV_STATE_DEV_STARTED;
    return (0);
}


/*
 * Scan PCI bus for L-CARD L7xx devices
 *
 * In: Device ptr
 * Out: 0  - ok
 *      -1 - error
 */
int scan_pci( l783m_t *l783m )
{
    struct pci_dev_info     dev_info;
    uint32_t                config_space[16];

    l783m->pci_attach_handle = pci_attach( 0 );
    if ( l783m->pci_attach_handle == -1 ) {
        MSG( STDF, "l783m: Unable to initialize PCI\n" );
        return (-1);
    }

    /* Initialize dev_info */
    memset( &dev_info, 0, sizeof( struct pci_dev_info ) );
    dev_info.VendorId = PCI_VENDOR_ID_PLX;
    dev_info.DeviceId = PCI_DEVICE_ID_PLX_PCI9030;

    /* Attach PLX9030 */
    l783m->pci_handle = pci_attach_device( NULL, PCI_INIT_ALL, l783m->descriptor.device_idx, &dev_info );
    if ( l783m->pci_handle == NULL ) {
        MSG( STDF, "l783m: Unable to locate PLX PCI9030\n" );
        return (-1);
    }

    /* Read PCI configuration space */
    memset( &config_space, 0, sizeof( uint32_t ) * 16 );
    if ( pci_read_config32( dev_info.BusNumber,
                            dev_info.DevFunc,
                            0,
                            16,
                            (char *)&config_space ) != PCI_SUCCESS ) {
        MSG( STDF, "l783m: Unable to read PCI cnfiguration space\n" );
        return (-1);
    }
    if ( l783m->verbose )
        MSG( STDF, "Device found: %X:%X\n", dev_info.VendorId, dev_info.DeviceId );
    l783m->board_info.DeviceID                  = config_space[0]  >> 16;
    l783m->board_info.VendorID                  = config_space[0]  & 0xff;
    l783m->board_info.StatusReg                 = config_space[1]  >> 16;
    l783m->board_info.CommandReg                = config_space[1]  & 0xffff;
    l783m->board_info.ClassCode                 = config_space[2]  >> 8;
    l783m->board_info.RevisionID                = config_space[2]  & 0xff;
    l783m->board_info.BIST                      = config_space[3]  >> 24;
    l783m->board_info.HeaderType                = (config_space[3] >> 16) & 0xff;
    l783m->board_info.LatencyTimer              = (config_space[3] >> 8) & 0xff;
    l783m->board_info.CacheLineSize             = config_space[3] & 0xff;
    l783m->board_info.BAR0                      = config_space[4];
    l783m->board_info.BAR1                      = config_space[5];
    l783m->board_info.BAR2                      = config_space[6];
    l783m->board_info.BAR3                      = config_space[7];
    l783m->board_info.BAR4                      = config_space[8];
    l783m->board_info.BAR5                      = config_space[9];
    l783m->board_info.CardbusCISpointer         = config_space[10];
    l783m->board_info.SubsystemID               = config_space[11]  >> 16;
    l783m->board_info.SubsystemVendorID         = config_space[11]  & 0xffff;
    l783m->board_info.ExpansionROMBaseAddress   = config_space[12];
    l783m->board_info.CapabilitiesPointer       = config_space[13]  & 0xff;
    l783m->board_info.Max_Lat                   = config_space[15]  >> 24;
    l783m->board_info.Min_Gnt                   = (config_space[15] >> 16) & 0xff;
    l783m->board_info.InterruptPin              = (config_space[15] >> 8) & 0xff;
    l783m->board_info.InterruptLine             = config_space[15]  & 0xff;
    if ( l783m->verbose > 2 ) {
        MSG( STDF, "PCI configuration space:\n" );
        MSG( STDF, "  Status Reg:                  0x%.2x\n", l783m->board_info.StatusReg );
        MSG( STDF, "  Command Reg:                 0x%.2x\n", l783m->board_info.CommandReg );
        MSG( STDF, "  Class Code:                  0x%.2x\n", l783m->board_info.ClassCode );
        MSG( STDF, "  Revision ID:                 0x%.2x\n", l783m->board_info.RevisionID );
        MSG( STDF, "  BIST:                        0x%.2x\n", l783m->board_info.BIST );
        MSG( STDF, "  Header Type:                 0x%.2x\n", l783m->board_info.HeaderType );
        MSG( STDF, "  Latency Timer:               0x%.2x\n", l783m->board_info.LatencyTimer );
        MSG( STDF, "  Cache Line Size:             0x%.2x\n", l783m->board_info.CacheLineSize );
        MSG( STDF, "  BAR0:                        0x%.2X\n", l783m->board_info.BAR0 );
        MSG( STDF, "  BAR1:                        0x%.2X\n", l783m->board_info.BAR1 );
        MSG( STDF, "  BAR2:                        0x%.2X\n", l783m->board_info.BAR2 );
        MSG( STDF, "  BAR3:                        0x%.2X\n", l783m->board_info.BAR3 );
        MSG( STDF, "  BAR4:                        0x%.2X\n", l783m->board_info.BAR4 );
        MSG( STDF, "  BAR5:                        0x%.2X\n", l783m->board_info.BAR5 );
        MSG( STDF, "  Cardbus CIS pointer:         0x%.2X\n", l783m->board_info.CardbusCISpointer );
        MSG( STDF, "  Subsystem ID:                0x%.2X\n", l783m->board_info.SubsystemID );
        MSG( STDF, "  Subsystem Vendor ID:         0x%.2X\n", l783m->board_info.SubsystemVendorID );
        MSG( STDF, "  Expansion ROM Base Address:  0x%.2X\n", l783m->board_info.ExpansionROMBaseAddress );
        MSG( STDF, "  Capabilities Pointer:        0x%.2X\n", l783m->board_info.CapabilitiesPointer );
        MSG( STDF, "  Max_Lat:                     0x%.2X\n", l783m->board_info.Max_Lat );
        MSG( STDF, "  Min_Gnt:                     0x%.2X\n", l783m->board_info.Min_Gnt );
        MSG( STDF, "  Interrupt Pin:               0x%.2X\n", l783m->board_info.InterruptPin );
        MSG( STDF, "  Interrupt Line:              0x%.2X\n", l783m->board_info.InterruptLine );
    }

    /* Check for board type */
    if ( l783m->board_info.SubsystemVendorID != PCI_SUB_VENDOR_ID_L783M ||
         l783m->board_info.SubsystemID != PCI_SUB_DEVICE_ID_L783M ) {
        MSG( STDF, "l783m: Unable to locate L-CARD L783M\n" );
        return (-1);
    }

    /* Config BOARD_INFO */
    l783m->board_info.bi.base_dsp_dm_offset = DSP_DM_START_ADDR;
    l783m->board_info.bi.ioplx = mmap_device_io( PLX_REG_LCR_EXTENT, PCI_IO_ADDR( l783m->board_info.BAR1 ) );
    l783m->board_info.bi.cntrl = l783m->board_info.bi.ioplx + PLX_REG_CNTRL;
    l783m->board_info.bi.eecs  = l783m->board_info.bi.ioplx + 0x54;
    l783m->board_info.bi.base  = mmap_device_io( IDMA_EXTENT, PCI_IO_ADDR( l783m->board_info.BAR2 ) );
    l783m->board_info.bi.array = mmap_device_memory( NULL, 0x1000, PROT_READ | PROT_WRITE | PROT_NOCACHE,
                                                     0, PCI_MEM_ADDR( l783m->board_info.BAR3 ) + 0x1000 );
    if ( l783m->board_info.bi.ioplx == MAP_DEVICE_FAILED ||
         l783m->board_info.bi.ioplx == MAP_DEVICE_FAILED ) {
        MSG( STDF, "l783m: Unable to mmap device I/O ports\n" );
        return (-1);
    }
    if ( l783m->board_info.bi.array == MAP_FAILED ) {
        MSG( STDF, "l783m: Unable to mmap device IDMA memory\n" );
        return (-1);
    }
    if ( l783m->verbose > 2 ) {
        MSG( STDF, "Memory & I/O:\n" );
        MSG( STDF, "  PLX control registers:       0x%X\n", l783m->board_info.bi.ioplx );
        MSG( STDF, "  PLX CNTRL control register:  0x%X\n", l783m->board_info.bi.cntrl );
        MSG( STDF, "  EEPROM Chip Select register: 0x%X\n", l783m->board_info.bi.eecs );
        MSG( STDF, "  I/O base address:            0x%X\n", l783m->board_info.bi.base );
        MSG( STDF, "  IDMA memory:                 0x%X\n", (uint32_t)l783m->board_info.bi.array );
    }
    eeprom_read_data( l783m, 0, (uint16_t *)&l783m->board_info.eeprom, sizeof( struct EEPROM_INFO ) / 2 );
    if ( l783m->verbose > 1 ) {
        MSG( STDF, "EEPROM:\n" );
        MSG( STDF, "  Serial:                      %s\n",  (char *)&l783m->board_info.eeprom.serial );
        MSG( STDF, "  Name:                        %s\n",  (char *)&l783m->board_info.eeprom.name );
        MSG( STDF, "  Rev.:                        %c\n",   l783m->board_info.eeprom.revision );
        MSG( STDF, "  DSP Type:                    %s\n",  (char *)&l783m->board_info.eeprom.dsp_type );
        MSG( STDF, "  Frequency:                   %li\n",  l783m->board_info.eeprom.freq );
        if ( l783m->board_info.eeprom.is_dac )
            MSG( STDF, "  DAC on board:                YES\n" );
        else
            MSG( STDF, "  DAC on board:                NO\n" );
    }

    plx_reset( l783m );

    return (0);
}


/*
 * Write DSP program memory
 *
 * In: Device ptr
 *     Start address
 *     Data ptr
 *     Data size (in int32_t units)
 * Out: 0  - ok
 *      -1 - error
 */
int write_dsp_pm( l783m_t *l783m, uint32_t addr, uint32_t *data, uint32_t size )
{
    uint32_t                c;

    if ( l783m->board_info.bi.array == NULL )
        return (-1);

    out16( l783m->board_info.bi.base + IDMA_ADDR, addr );

    while ( size > 0 ) {

        c = 1024;
        if ( size < c )
            c = size;

        memcpy( l783m->board_info.bi.array, data, c * sizeof(uint32_t) );
        data = (uint32_t *)((uint32_t)data +  c * sizeof(uint32_t));
        size -= c;

    }

    return (0);
}


/*
 * Write DSP data memory
 *
 * In: Device ptr
 *     Start address
 *     Data ptr
 *     Data size (in uint16_t units)
 * Out: 0  - ok
 *      -1 - error
 */
int write_dsp_dm( l783m_t *l783m, uint32_t addr, uint16_t *data, uint32_t size )
{
    uint32_t                c;

    if ( l783m->board_info.bi.array == NULL )
        return (-1);

    addr = (addr & 0x3FFF) | 0x4000;
    out16( l783m->board_info.bi.base + IDMA_ADDR, addr );

    while ( size > 0) {

        c = 2048;
        if ( size < c )
            c = size;

        memcpy( l783m->board_info.bi.array, data, c * sizeof(uint16_t) );
        data = (uint16_t *)((uint32_t)data +  c * sizeof(uint16_t));
        size -= c;

    }

    return (0);
}


/*
 * Read DSP data memory
 *
 * In: Device ptr
 *     Start address
 *     Data ptr
 *     Data size (in uint16_t units)
 * Out: 0  - ok
 *      -1 - error
 */
int read_dsp_dm( l783m_t *l783m, uint32_t addr, uint16_t *data, uint32_t size )
{
    uint32_t                c;

    if ( l783m->board_info.bi.array == NULL )
        return (-1);

    addr = (addr & 0x3FFF) | 0x4000;
    out16( l783m->board_info.bi.base + IDMA_ADDR, addr );

    while ( size > 0) {

        c = 2048;
        if ( size < c )
            c = size;

        copy_32( (uint8_t *)data, (uint8_t *)l783m->board_info.bi.array, c * 2 );
        data = (uint16_t *)((uint32_t)data +  c * sizeof(uint16_t));
        size -= c;

    }

    return (0);
}


/*
 * Read from DSP data memory (1 word)
 *
 * In: Device ptr
 *     Address
 * Out: x    - value
 *      (-1) - error
 */
int read_dsp_dm_word( l783m_t *l783m, uint32_t addr )
{
    int                     prev_idma_addr;
    int                     x = 0x0;

    /* Save IDMA address */
    prev_idma_addr = l783m->idma_addr;

    addr = (addr & 0x3FFF) | 0x4000;
    l783m->idma_addr = addr;    /* if IRQ occur */
    out16( l783m->board_info.bi.base + IDMA_ADDR, addr );
    x = (int)(*(uint16_t *)l783m->board_info.bi.array);

    /* Restore IDMA address */
    l783m->idma_addr = prev_idma_addr;
    if ( prev_idma_addr >= 0 )
        out16( l783m->board_info.bi.base + IDMA_ADDR, prev_idma_addr );

    return (x);
}


/*
 * Write to DSP data memory (1 word)
 *
 * In: Device ptr
 *     Address
 *     Value
 */
void write_dsp_dm_word( l783m_t *l783m, uint32_t addr, uint16_t val )
{
    int                     prev_idma_addr;

    /* Save IDMA address */
    prev_idma_addr = l783m->idma_addr;

    addr = (addr & 0x3FFF) | 0x4000;
    l783m->idma_addr = addr;    /* if IRQ occur */
    out16( l783m->board_info.bi.base + IDMA_ADDR, addr );
    *(uint16_t *)l783m->board_info.bi.array = val;

    /* Restore IDMA address */
    l783m->idma_addr = prev_idma_addr;
    if ( prev_idma_addr >= 0 )
        out16( l783m->board_info.bi.base + IDMA_ADDR, prev_idma_addr );
}


/*
 * Execute DSP BIOS command
 *
 * In: Device ptr
 *     Command
 *     Unblock thread immediately (0 is returned)
 * Out: 0  - ok
 *      -1 - error
 */
int dsp_bios_command( l783m_t *l783m, uint16_t command, uint8_t unblock )
{
    int status;
    int i;

    write_dsp_dm_word( l783m, BIOS_VARIABLE_COMMAND + l783m->board_info.bi.base_dsp_dm_offset, command );

    out16( l783m->board_info.bi.base + DSP_INTR, 0 );

    for(i = 0; i < 300; i++) {

        status = read_dsp_dm_word( l783m, BIOS_VARIABLE_COMMAND + l783m->board_info.bi.base_dsp_dm_offset );
        if ( status == 0 )
            return (0);
        if ( status == -1 )
            break;
        if ( unblock )
            return (0);
        //usleep( 1 );

    }

    return (-1);
}


/*
 * Set DSP ADC buffer
 *
 * In: Device ptr
 *     Size
 *     fragmentation size
 * Out: 0  - ok
 *      -1 - error
 */
int dsp_adc_set_buffer( l783m_t *l783m, uint32_t size, uint32_t fsize )
{
    int                     b_wd = size / sizeof( uint16_t );
    int                     f_wd = fsize / sizeof( uint16_t );

    if ( f_wd == 0 )
        return (-1);

    if ( f_wd * 2 > l783m->adc_fifo_max_sz )
        f_wd = l783m->adc_fifo_max_sz / 2;
    if ( b_wd > l783m->adc_fifo_max_sz )
        b_wd = l783m->adc_fifo_max_sz;
    if ( b_wd == 0 )
        b_wd = (int)(l783m->adc_fifo_max_sz / f_wd) * f_wd;

    write_dsp_dm_word( l783m, BIOS_VARIABLE_IRQ_STEP + l783m->board_info.bi.base_dsp_dm_offset, f_wd );
    write_dsp_dm_word( l783m, BIOS_VARIABLE_ADC_NEW_FIFO_LENGTH + l783m->board_info.bi.base_dsp_dm_offset, b_wd );

    return dsp_bios_command( l783m, BIOS_COMMAND_ADC_FIFO_CFG, 0 );
}


/*
 * Set DSP ADC timing
 *
 * In: Device ptr
 *     Frame rate
 *     Channel rate
 * Out: 0  - ok
 *      -1 - error
 */
int dsp_adc_set_timing( l783m_t *l783m, int *fr, int *cr )
{
    long                     ch_delay      = 0;  /* Channel delay */
    long                     iframe_delay  = 0;  /* Inter-frame delay */
    long                     fst_smp_delay = 0;  /* First sample delay */
    long                     tmp           = 0;

    tmp = l783m->board_info.eeprom.freq;

    if ( *cr > ADC_MAX_FREQ || *cr <= 0 )
        *cr = ADC_MAX_FREQ;
    ch_delay = tmp / *cr;
    if ( ch_delay < 1 )
        ch_delay = 1;
    if ( ch_delay > 0xffff )
        ch_delay = 0xffff;
    *cr = tmp / ch_delay;

    if ( fr == NULL )
        return (-1);
    if ( *fr == 0 || *cr > ADC_MAX_FREQ )
        iframe_delay = 0;                           /* No iframe_delay if DSP ADC rate > 1MHz */
    else {
        iframe_delay = tmp / *fr;                   /* T - Frame period */
        iframe_delay -= ch_delay * l783m->channels; /* Inter-frame delay */
        iframe_delay /= ch_delay;                   /* Inter-frame delay (in channel delay units) */
    }
    if ( iframe_delay < 0 )
        iframe_delay = 0;
    if ( iframe_delay > 0xffff )
        iframe_delay = 0xffff;
    *fr = tmp / (ch_delay * (l783m->channels + iframe_delay));
    fst_smp_delay = ch_delay * 2 + 5;
    ch_delay -= 1;

    if ( l783m->verbose > 3 ) {
        MSG( STDF, "DSP clock: %ld Hz\n", tmp );
        MSG( STDF, "ADC rate: %ld\nInter-frame delay: %ld\nFirst sample delay: %ld\n", 
                   ch_delay, iframe_delay, fst_smp_delay );
    }

    /* Apply */
    write_dsp_dm_word( l783m, BIOS_VARIABLE_ADC_RATE       + l783m->board_info.bi.base_dsp_dm_offset, ch_delay );
    write_dsp_dm_word( l783m, BIOS_VARIABLE_ADC_INTERFRAME + l783m->board_info.bi.base_dsp_dm_offset, iframe_delay );
    write_dsp_dm_word( l783m, BIOS_VARIABLE_FIRST_SAMPLE   + l783m->board_info.bi.base_dsp_dm_offset, fst_smp_delay );

    return dsp_bios_command( l783m, BIOS_COMMAND_SET_ADC_FRAME, 0 );
}
