//	Rhapsody		: 7.1 
//	Component		: oxfFiles 
//	Configuration 	: generic
//	Model Element	: OMHeap
//!	File name		: $Source: R:/StmOO/Master/cg/LangCpp/oxf/rcs/omheap.h $
//!	File version	: $Revision: 1.40 $
//
//!	Date changed	: $Date: 2007/04/06 07:08:08 $
//!	Last change by	: $Author: ilgiga $
//
//	(c) Copyright Telelogic 2004, 2007
//



#ifndef omheap_H 

#define omheap_H 

#include "OXFMemoryManagerMacros.h"
#include "OXFNotifyMacros.h"
#include "OMAbstractContainer.h"
#include "ommemorymanager.h"

//----------------------------------------------------------------------------
// omheap.h                                                                  
//----------------------------------------------------------------------------

template <class  Concept> class OMIterator;

//## package Design::oxf::Services::Containers::TypeSafe 


// //////////////////////////////////////////////////////////
// template class OMHeap
// 
// The Heap invariants:
// 1. theHeap[0] is empty - this is so we have "easy arithmetic"
// 2. theHeap[1] -- theHeap[count] hold the actual elements
// 3. theHeap[i]<theHeap[2*i] && theHeap[i]<theHeap[2*i+1]
// 			  for all i>0, 2*i<=count, 2*i+1<=count.
// 	  Hence Min = theHeap[1]
// 
// 
// OMHeap<Node> is a heap holding elements of type "Node*"
// 
// //////////////////////////////////////////////////////////
//## class OMHeap 
template <class Node> class OMHeap : public OMAbstractContainer<Node> {


////    Constructors and destructors    ////
public :
    
    // Initialize a heap with a given size
    // Argument int size : 
    // The heap size (number of elements)
    //## operation OMHeap(int) 
    OMHeap(int size = 100) : count(0) ,heapSize(0) ,theheap(NULL) {
        //#[ operation OMHeap(int) 
        heapSize = size + 1; // theheap[0] is never used
        theheap = OMNEW(Node*, heapSize);
        //#]
    }
    
    
    // copy constructor and assignment operator
    // Argument const OMHeap<Node> & h : 
    // The heap to copy
    //## operation OMHeap(const OMHeap<Node> & ) 
    OMHeap(const OMHeap<Node> & h) : count(0) ,heapSize(0) ,theheap(NULL) {
        //#[ operation OMHeap(const OMHeap<Node> & ) 
        copy(h);
        //#]
    }
    
    
    // cleanup
    //## operation ~OMHeap() 
    ~OMHeap() {
        //#[ operation ~OMHeap() 
        OMDELETE(theheap, sizeof(Node*) * heapSize);
        theheap = NULL;
        //#]
    }
    


////    Operations    ////
public :
    
    // Add e to heap
    // Argument Node* e : 
    // The element to add
    //## operation add(Node*) 
    inline bool add(Node* e) {
        //#[ operation add(Node*) 
        bool res = true;
        if (count + 1 < heapSize)
        {
        	++count;
        	takeUp(e,count);
        }
        else
        {
        	OM_NOTIFY_TO_ERROR("\n\nHeap overflow\n\n");
        	res = false;
        }
        return res;
        //#]
    }
    
    
    // Return the position of the first e such that 
    // (*e) == (*clone) return 0 if not found
    // Argument Node* clone : 
    // The search prototype
    //## operation find(Node*) const 
    inline int find(Node* clone) const {
        //#[ operation find(Node*) const 
        // Return the position of the first e such that 
        // (*e) == (*clone) return 0 if not found
        for (int i=1;i<=count;i++)
        	if (*(theheap[i]) == *clone) return i;
        	return 0;
        //#]
    }
    
    
    // Check if the heap is empty
    //## operation isEmpty() const 
    inline int isEmpty() const {
        //#[ operation isEmpty() const 
        return count == 0;
        //#]
    }
    
    
    // Assignment operator
    // Argument const OMHeap<Node> & h : 
    // The heap to copy
    //## operation operator=(const OMHeap<Node> & ) 
    inline OMHeap<Node> &  operator=(const OMHeap<Node> & h) {
        //#[ operation operator=(const OMHeap<Node> & ) 
        copy(h);
        return *this;
        //#]
    }
    
    
    // Remove the first e such that
    // (*e) == (*clone)	
    // return e if found NULL otherwise
    // Argument Node* clone : 
    // The element to remove
    //## operation remove(Node*) 
    inline Node *  remove(Node* clone) {
        //#[ operation remove(Node*) 
        int position = find(clone);
        if (position==0)
        	return NULL;
        
        Node* retVal = theheap[position];
        // Remove theheap[position] by:
        // 1. Decrementing count, 2. Making position empty and
        // 3. Trying to place in it the last heap element
        Node* last = theheap[count]; // 3.
        count--;					 // 2.
        // Reorder the heap             1.
        if (position == 1 || (*last > *theheap[position/2])) {
        	takeDown(last,position);
        }
        else {
        	takeUp(last,position);
        }
        return retVal; 
        //#]
    }
    
    
    // Cleanup the heap
    //## operation removeAll() 
    inline void removeAll() {
        //#[ operation removeAll() 
        while (count > 0) {
        	trim();
        }
        //#]
    }
    
    
    // Get the top of the heap
    //## operation top() const 
    inline Node *  top() const {
        //#[ operation top() const 
        return theheap[1];
        //#]
    }
    
    
    // Remove top
    //## operation trim() 
    inline void trim() {
        //#[ operation trim() 
        count--;
        if (count>0) {
        	takeDown(theheap[count+1], 1);  
        }
        //#]
    }
    

private :
    
    // Copy a heap
    // Argument const OMHeap<Node> & h : 
    // The heap to copy
    //## operation copy(const OMHeap<Node> & ) 
    inline void copy(const OMHeap<Node> & h) {
        //#[ operation copy(const OMHeap<Node> & ) 
        if (theheap != h.theheap) {
        	// prevent self copy
        	if (theheap != NULL) {
        		// cleanup before copy
        		OMDELETE(theheap, sizeof(Node*) * heapSize);
        	}
        	count = h.count;
        	heapSize = h.heapSize;
        	theheap = OMNEW(Node*, heapSize);
        	for (int i = 1; i <= count; ++i) {
        		// theheap[0] is unused
        		theheap[i] = h.theheap[i];
        	}
        }
        //#]
    }
    
    
    // Get the element at the given position (called by the iterator)
    // Argument void * pos : 
    // The iterator position
    //## operation getCurrent(void *) const 
    inline virtual Node&  getCurrent(void * pos) const {
        //#[ operation getCurrent(void *) const 
        if ((int)pos >= 1 && (int)pos <= count) {
        	return *(theheap[(int)pos]);
        }
        else {
        	return OMNullValue<Node>::get();
        }
        //#]
    }
    
    
    // Set the initial position for the iterator
    // Argument void * & pos : 
    // The initial position (out)
    //## operation getFirst(void * & ) const 
    inline virtual void getFirst(void * & pos) const {
        //#[ operation getFirst(void * & ) const 
        pos = reinterpret_cast<void*>(1);
        //#]
    }
    
    
    // Update the provided position to the next position in the container
    // Argument void * & pos : 
    // The position to advance
    //## operation getNext(void * & ) const 
    inline virtual void getNext(void * & pos) const {
        //#[ operation getNext(void * & ) const 
        pos = reinterpret_cast<void*>(
        	reinterpret_cast<int>(pos) + 1);
        //#]
    }
    
    
    // e is a node which needs to be placed in the heap.
    // Its position can be either 'emptyPos' which is currently empty or some position lower than emptyPos (if 
    // elements currently there are smaller than e(
    // Argument Node* e : 
    // The element to move
    // Argument int emptyPos : 
    // The next empty position
    //## operation takeDown(Node*,int) 
    inline void takeDown(Node* e, int emptyPos) {
        //#[ operation takeDown(Node*,int) 
        int oneLower = emptyPos*2;
        // Case 1 - Currently there are no lower positions
        if (oneLower > count) { // Place e in empty position and quit
        	theheap[emptyPos] = e;
        	return;
        }
        
        // Case 2 - There is one lower position
        if (oneLower == count) {
        	if (*theheap[oneLower] < *e) { // The element there is smaller
        		// Put the smaller element in the empty position
        		theheap[emptyPos] = theheap[oneLower];
        		// Put e in the lower position (which we know does not
        		// have occupied positions below it)
        		theheap[oneLower] = e;
            } else  // The element there is bigger
        		// Put e in the empty position
        		theheap[emptyPos]= e ;
        	return;
        }
        
        // The general Case There are two lower positions
        // (oneLower and oneLower+1)
        
        // Make oneLower the position holding the smaller of the items
        // Since theHeap[i]<theHeap[2*i] && theHeap[i]<theHeap[2*i+1]
        // is our invariant
        if (*theheap[oneLower]>*theheap[oneLower+1]) 
        	oneLower++;
        
        if (*theheap[oneLower]<*e) { // The "lower" element is smaller
        	// Put the smaller element in the empty position
        	theheap[emptyPos]=theheap[oneLower];
        	takeDown(e,oneLower);
        } else
        	// Put e in the empty position
        	theheap[emptyPos] = e ;
        //#]
    }
    
    
    // e is a node which needs to be placed in the heap.
    // Its position can be either 'emptyPos' which is currently empty or some position higher than emptyPos (if 
    // elements currently there are bigger than e)
    // Argument Node* e : 
    // The element to move
    // Argument int emptyPos : 
    // The next empty position
    //## operation takeUp(Node*,int) 
    inline void takeUp(Node* e, int emptyPos) {
        //#[ operation takeUp(Node*,int) 
        for (;;)
        {
        	int oneHigher = emptyPos/2;
        	if (oneHigher > 0 && (*theheap[oneHigher] > *e)) {
        		// There is a higher position and 
        		// the element there is bigger than e
        		// 1. Put the bigger element in the empty position
        		theheap[emptyPos]=theheap[oneHigher];
        		// 2. Make its old position the empty one
        		emptyPos = oneHigher;
        	} else {
        		// Put e in the empy position
        		theheap[emptyPos] = e;
        		return;
        	}
        }
        //#]
    }
    


////    Additional operations    ////
public :
    
    //## auto_generated 
    inline int getCount() const {
        return count;
    }
    
    
    //## auto_generated 
    inline int getHeapSize() const {
        return heapSize;
    }
    


////    Attributes    ////
private :
    
    // The number of items currently in the Heap
    int count;		//## attribute count 
    
    // The memory allocated for theHeap
    int heapSize;		//## attribute heapSize 
    
    // An array of Node*'s - the data of the Heap
    Node** theheap;		//## attribute theheap 
    

};
//## class OMHeap 





#endif  
//
//! Log: $Log: omheap.h $
//! Log: Revision 1.40  2007/04/06 07:08:08  ilgiga
//! Log: bug fix 98191
//


